#!/bin/sh

# Defines a function that will be called if a user who is
# connected via SSH leaves the session (Types 'exit' or ctrl+d)

pf_check_unstaged() {
    # Check if unstaged changes are present in a repository
    # First argument: Path to repository
    # Second argument: Use "sudo" if given

    # Valid path must be given
    [ -d $1 ] || return

    # Use "sudo" with GIT if wanted.
    # Used for sudo owned repositories like "/etc/"
    GIT="git"
    [ "$2" = "sudo=true" ] && GIT="sudo git"

    # Check if unstaged changes are present
    UNSTAGED=$(cd $1; $GIT status --porcelain=v1)
    # Return if git command was not successful
    [ $? -ne 0 ] && return 1

    # Warn if unstaged changes are present
    if [ -n "$UNSTAGED" ]; then
        echo "WARNING: Unstaged changes left in ${1}:"
        echo "$UNSTAGED"
        printf "\n"
    fi
}

pf_onexit() {
    # This function will be called if a SSH session is closed

    # Check repositories for unstaged changes
    pf_check_unstaged "/opt/perfact/dbutils-zoperepo/"
    pf_check_unstaged "/etc/" "sudo=true"
}

# Call the pf_onexit function if a SSH session closes
# Checking for $PS1 makes sure that we are in an interactive shell
[ -n "$SSH_CONNECTION" ] && [ -n "$PS1" ] && trap pf_onexit EXIT
