# Provide python2 compatible versions of 3.x methods in module "tempfile"

from tempfile import mkdtemp
import weakref as _weakref
import os as _os
import shutil as _shutil


_rmtree = _shutil.rmtree


class TemporaryDirectory(object):
    """Create and return a temporary directory.  This has the same
    behavior as mkdtemp but can be used as a context manager.  For
    example:

        with TemporaryDirectory() as tmpdir:
            ...

    Upon exiting the context, the directory and everything contained
    in it are removed.
    """

    def __init__(self, suffix='', prefix='tmp', dir=None):
        self.name = mkdtemp(suffix, prefix, dir)
        _weakref.proxy(self, self._cleanup)

    @classmethod
    def _rmtree(cls, name):
        def onerror(func, path, exc_info):
            if issubclass(exc_info[0], IOError):
                def resetperms(path):
                    try:
                        _os.chflags(path, 0)
                    except AttributeError:
                        pass
                    _os.chmod(path, 0o700)

                try:
                    if path != name:
                        resetperms(_os.path.dirname(path))
                    resetperms(path)

                    try:
                        _os.unlink(path)
                    # PermissionError is raised on FreeBSD for directories
                    except IOError:
                        cls._rmtree(path)
                except IOError:
                    pass
            elif issubclass(exc_info[0], IOError):
                pass
            else:
                raise

        _rmtree(name, onerror=onerror)

    def _cleanup(self):
        self._rmtree(self.name)

    def __repr__(self):
        return "<{} {!r}>".format(self.__class__.__name__, self.name)

    def __enter__(self):
        return self.name

    def __exit__(self, exc, value, tb):
        self.cleanup()

    def cleanup(self):
        self._rmtree(self.name)
