#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# datasourceProvider.py  -  building datasource-connections for dbcached.py
#
# $Revision: 2.10 $
#
# Copyright (C) 2015 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

''' This is the successor of dbconnector.py
Today the data sources for dbcached.py can not only be databases.
So this is a module without the database stuff.
'''

from .dbdatasources import ConnectionDecorator

from .SoapPseudoConnection import SoapPseudoConnection
from .ODataPseudoConnection import ODataPseudoConnection
from . import RestConnections
from . import saprfc
import importlib


def createDS(dbtype=None, connectstring=None, cacheconn_id=None,
             config=None, classname=None):
    ''' Connect a datasource of a given type with the given connectstring '''

    dcon = None

    if config and classname:
        # Use new DataSource-Classes
        classname_parts = classname.split('.')
        py_path = '.'.join(classname_parts[:-1])
        classname_end = classname_parts[-1]

        try:
            module = importlib.import_module(
                'perfact.daemondatasources.providers.'+py_path)
            target_class = getattr(module, classname_end)
        except AttributeError as e:
            print("No datasource-class found for '{classname}'".format(
                classname=classname,
            ))
            raise e

        dcon = target_class(classname=classname, **config)
    elif dbtype and connectstring:
        # Use old DataSource-Classes
        # This list should not be appended anymore.
        # Consider migrating datasources to the new format instead

        if dbtype == 'soap':
            dcon = SoapPseudoConnection(connector=connectstring,
                                        dbtype=dbtype)
        elif dbtype == 'soap1.2':
            dcon = SoapPseudoConnection(connector=connectstring,
                                        dbtype=dbtype,
                                        version='1.2')
        elif dbtype == 'soap1.1':
            dcon = SoapPseudoConnection(connector=connectstring,
                                        dbtype=dbtype,
                                        version='1.1')
        elif dbtype == 'soap1.2_ntlm':
            dcon = SoapPseudoConnection(connector=connectstring,
                                        dbtype=dbtype,
                                        version='1.2',
                                        authtype='ntlm')
        elif dbtype == 'soap1.1_ntlm':
            dcon = SoapPseudoConnection(connector=connectstring,
                                        dbtype=dbtype,
                                        version='1.1',
                                        authtype='ntlm')
        elif dbtype in ['odata', 'odata_token', 'odata_token_patch',
                        'odata_noauth']:
            dcon = ODataPseudoConnection(connector=connectstring,
                                         dbtype=dbtype)
        elif dbtype == 'restsession':
            dcon = RestConnections.SessionConnection(connector=connectstring,
                                                     dbtype=dbtype)
        elif dbtype == 'saprfc':
            dcon = saprfc.SAPConnection(connector=connectstring,
                                        dbtype=dbtype)
        elif dbtype == 'rest':
            dcon = RestConnections.RestPseudoConnection(
                connector=connectstring,
                dbtype=dbtype)
        else:
            try:
                dcon = ConnectionDecorator(connector=connectstring,
                                           dbtype=dbtype,
                                           cacheconn_id=cacheconn_id)
            except NotImplementedError:
                mes = ('Encountered unknown connection type %s '
                       '- this cannot be used!') % (dbtype)
                raise dsException(
                    dbtype,
                    'cacheconn_id={}'.format(cacheconn_id),
                    mes
                )
    else:
        raise Exception("Invalid combination of createDS arguments")

    return dcon


class dsException(Exception):
    def __init__(self, *args):
        self.args = args
        dbtype, connection, error = args
        self.value = 'Connection failed for %s with %s (%s)'\
                     % (dbtype, connection, error)
