from . import ProviderInterface
from os.path import isfile
import requests


class OData(ProviderInterface):
    def __init__(self, classname, base_url, csrf=None, certs=None):
        super(ProviderInterface, self).__init__()

        self.classname = classname
        self.session = None
        self.base_url = base_url
        self.csrf = csrf or {}
        self.certs = certs or []

        for cert in self.certs:
            assert isfile(cert), 'Unable to find certificate at {cert}'.format(
                cert=cert
            )

    def connect(self, reconnect=False):
        if not self.connected or reconnect:
            self.disconnect()
            self.session = requests.Session()

    def disconnect(self):
        self.session = None

    def close(self):
        self.disconnect()

    def _is_csrf_configured(self):
        """ Check if this connection should use CSRF-Validation based on
        given config values.
        """
        csrf_config_keys = [
            'token_fetch_url',
            'token_header',
            'token_cmd',
        ]

        # Check if all required keys available and are set to a not-None Value
        # Considering nill-strings to be not-None here
        config_values = [
            (self.csrf.get(csrf_key) is not None) for csrf_key
            in csrf_config_keys
        ]
        return all(config_values)

    def _fetch_csrf_token(self):
        """ Fetch a CSRF-Token based on given csrf config.
        Let's assume some values for this examples sake.
        fetch_url: 'http:localhost/sub/domain'
        token_cmd: 'fetch'
        header: 'x-csrf-token'

        We send a get request to the fetch url where the request's header
        'x-csrf-token' is set to the value 'fetch'.
        The endpoint will set two headers in the response
        'set-cookie' and 'x-csrf-token'.
        set-cookie is handled by requests (This is why we are using a session)
        x-csrf-token will contain the desired token.
        We yoink this and return it.

        In order to pass the csrf-validation the operation which
        triggered the fetch needs to have the cookie and have this
        token set in it's header as 'x-csrf-token'
        """
        headers = {
            self.csrf['token_header']: self.csrf['token_cmd'],
        }

        res = self.session.get(
            url=self.csrf['token_fetch_url'],
            headers=headers,
            cert=self.certs,
        )

        assert self.csrf['token_header'] in res.headers, \
            'No CSRF token received after fetch'

        return res.headers[self.csrf['token_header']]

    def read(self, query, add_headers=None, **kwargs):
        self.connect(reconnect=False)
        target_url = '{base_url}/{query}'.format(
            base_url=self.base_url,
            query=query,
        )

        headers = {'Accept': 'application/json'}
        if add_headers:
            headers.update(add_headers)

        if self._is_csrf_configured():
            csrf_token = self._fetch_csrf_token()
            headers[self.csrf['token_header']] = csrf_token

        raw_result = self.session.get(
            url=target_url,
            headers=headers,
            cert=self.certs,
        )
        return raw_result.json()

    def write(self, data, add_headers=None, method='post', **kwargs):
        self.connect(reconnect=False)
        headers = {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
        }
        if add_headers:
            headers.update(add_headers)

        if self._is_csrf_configured():
            csrf_token = self._fetch_csrf_token()
            headers[self.csrf['token_header']] = csrf_token

        write_method = getattr(self.session, method)
        res = write_method(
            url=self.base_url,
            headers=headers,
            data=data,
            cert=self.certs,
        )

        return res.json()

    @property
    def connected(self):
        return self.session is not None
