#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# dbdaemontools.py  -  Tools for dbbookd and dbcached
#
# $Revision: 1.0 $
#
# Copyright (C) 2017 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
# USA.
#

import time
import select
import psycopg2
import psycopg2.sql
from . import dbconn

dbcached_notifier = 'dbcache_done'


def bookNow(xfertype_id=None, localid='',
            waitfor=5, username='missing username',
            dryrun=False, connectstring=None):
    '''Alias for book_now().
    '''
    return book_now(xfertype_id=xfertype_id, localid=localid,
                    waitfor=waitfor, username=username,
                    dryrun=dryrun, connectstring=connectstring)


def book_now(xfertype_id=None, localid='',
             waitfor=5, username='missing username',
             dryrun=False, connectstring=None):
    '''Perform an xferqueue booking synchronously.

    Test: dummy booking type must exists, in order to perform the
    round trip.
    '''
    # check if we need to switch to the test-db
    if connectstring:
        conn = custom_connection(connectstring)
    else:
        conn = dbconn.DBConn()
        conn.connect()
    # insert and get the xferqueue_id
    xferqueue_id = insert_xferqueue(
        username=username, conn=conn,
        xfertype_id=xfertype_id, localid=localid)
    # check until done or timeout
    lc = 1
    start_time = time.time()
    while (lc < 3):
        now_time = time.time()
        seconds_run = (now_time - start_time)
        seconds_togo = int(waitfor - seconds_run)
        print('waitfor: %s seconds_run: %s seconds_togo: %s lc is: %s' %
              (str(waitfor), str(seconds_run), str(seconds_togo), str(lc)))
        if seconds_togo > 0:
            # still time left
            # wait for notify
            time.sleep(0.1)
            lc = get_xferstatus(xferqueue_id, conn)
        else:
            # time is up
            lc = get_xferstatus(xferqueue_id, conn)
            if (lc == 1):
                # dbbookd not responding
                lc = 99
            if (lc == 2):
                # remote system not responding
                lc = 100
    if lc == 99:
        # set to error in case of timeout
        neutralize_xfer(xferqueue_id, conn)
        lc = 4
    if lc == 100:
        lc = 2
        # we can not rollback the booking and can not wait for the
        # remote system
        raise Exception('remote system not responding')

    # return to caller
    if connectstring:
        # disconnect custom_connection
        conn.close()
    return {'lc': lc, 'xferqueue_id': xferqueue_id}


def insert_xferqueue(username, conn, xfertype_id=None, localid=''):
    ''' Simply insert into xferqueue and return the xferqueue_id'''
    q = """ insert into xferqueue (
xferqueue_author,
xferqueue_xfertype_id,
xferqueue_localid)
values ( '%s',%s,%s) returning xferqueue_id
""" % (username, xfertype_id, localid)
    res = query_q(conn=conn, query=q)
    if res and len(res):
        return str(res[0][0])
    else:
        raise AssertionError('Could not insert bookingrequest!')


def get_xferstatus(xferqueue_id, conn):
    ''' get the numeric status of the given xfer '''
    if not xferqueue_id:
        raise ValueError('No xferqueue_id')
    q = ("select xferqueue_xferstatus_id from xferqueue "
         "where xferqueue_id = " + str(int(xferqueue_id)))
    res = query_q(conn=conn, query=q)
    if res and len(res):
        return int(res[0][0])
    else:
        raise AssertionError('Could not request status!')


def neutralize_xfer(xferqueue_id, conn):
    q = ("update xferqueue set "
         "  xferqueue_xferstatus_id = 4, "
         "  xferqueue_error = 'time out for instant booking, "
         "dbbookd not responding' "
         "where xferqueue_xferstatus_id = 1 "
         "  and xferqueue_id = " + str(int(xferqueue_id)))
    execute_q(conn=conn, query=q)


def refreshCacheNow(cachestat_id=None, params='',
                    waitfor=5, tablename=None,
                    username='missing username',
                    dryrun=False, connectstring=None):
    '''Alias for refresh_cache_now().
    '''
    return refresh_cache_now(
        cachestat_id=cachestat_id,
        params=params,
        waitfor=waitfor,
        tablename=tablename,
        username=username,
        dryrun=dryrun,
        connectstring=connectstring,
    )


def refresh_cache_now(cachestat_id=None, params='',
                      waitfor=5, tablename=None,
                      username='missing username',
                      dryrun=False, connectstring=None):
    '''Refresh a cache through cachereq synchronously.
    '''
    # check if we need to switch to the test-db
    if connectstring:
        conn = custom_connection(connectstring)
    else:
        conn = dbconn.DBConn()
        conn = conn.connect()
    # insert and get the cachereq_id
    cachereq_id = insert_cachereq(
        username=username,
        conn=conn,
        cachestat_id=cachestat_id,
        params=params,
        executein=0,
        tablename=tablename,
    )
    activate_notifier(conn, dbcached_notifier)
    # check until done or timeout
    lc = 1
    start_time = time.time()
    while (lc < 3):
        now_time = time.time()
        seconds_run = (now_time - start_time)
        seconds_togo = int(waitfor - seconds_run)
        if seconds_togo > 0:
            # still time left
            # wait for notify
            fd = conn.fileno()
            readable, writable, exceptions = select.select(
                [fd], [], [], seconds_togo)
            notifies(dbcached_notifier, conn)
            lc = get_cachereqlc(cachereq_id, conn)
        else:
            # time is up
            lc = get_cachereqlc(cachereq_id, conn)
            if (lc < 3):
                lc = 99
    if lc == 99:
        # set to error in case of timeout
        neutralize_cachereq(cachereq_id, conn)
        lc = 4
    # return to caller

    if connectstring:
        # disconnect custom_connection
        conn.close()
    return {'lc': lc, 'cachereq_id': cachereq_id}


def insert_cachereq(username, conn, cachestat_id=None, params='',
                    executein=0, tablename=None):
    ''' Simply insert a cachereq and return the cachereq_id'''
    if (not cachestat_id):
        if tablename:
            cachestat_id = psycopg2.sql.SQL(
                """(select cachestat_id from cachestat """
                """where cachestat_tablename = {})""").format(
                    psycopg2.sql.Literal(tablename)
                )
        else:
            raise ValueError('No id and no tablename!')
    else:
        cachestat_id = psycopg2.sql.Literal(cachestat_id)
    # check cachestat (could be switched off for better performance)
    check = psycopg2.sql.SQL(
        """ select cachestat_id from cachestat """
        """where cachestat_id = {}""").format(cachestat_id)
    checkres = query_q(conn=conn, query=check)
    if not checkres:
        msg = 'Query for cachestat defect: %s' % check
        raise ValueError(msg)
    if not len(checkres):
        msg = 'No cachestat found per check: "%s"' % check
        raise ValueError(msg)
    cachestat_id = str(checkres[0][0])

    if not params:
        params = ''
    else:
        params = params.replace("'", "''")
    q = """ insert into cachereq (
cachereq_author,
cachereq_cachestat_id,
cachereq_executein,
cachereq_params)
values ( '%s',%s,%s,'%s'
) returning cachereq_id
""" % (username, cachestat_id, executein, params)
    res = query_q(conn=conn, query=q)
    if res and len(res):
        return str(res[0][0])
    else:
        raise AssertionError('Could not insert request!')


def get_cachereqlc(cachereq_id, conn):
    ''' get the numeric lc of the given cachereq '''
    if not cachereq_id:
        raise ValueError('No cachereq_id')
    q = (
        "select cachereq_cachereqlc_id from cachereq "
        "where cachereq_id = " + str(int(cachereq_id))
    )
    res = query_q(conn=conn, query=q)
    if res and len(res):
        return int(res[0][0])
    else:
        raise AssertionError('Could not request LC!')


def neutralize_cachereq(cachereq_id, conn):
    q = (
        "update cachereq set cachereq_cachereqlc_id = 4, "
        "cachereq_error = 'time out for instant refresh' "
        "where cachereq_cachereqlc_id = 1 "
        "  and cachereq_id = " + str(int(cachereq_id))
    )
    execute_q(conn=conn, query=q)


def notifies(notifier, conn):
    '''Fetch notifies.'''
    cur = conn.cursor()
    cur.execute('select 1')
    conn.commit()
    n = cur.connection.notifies
    n = list([a for a in [a[0] for a in n] if a == notifier])
    cur.close()
    return len(n)


def activate_notifier(conn, notifier):
    q = 'listen ' + notifier
    execute_q(conn, q)


def custom_connection(connectstring):
    local_conn = psycopg2.connect(connectstring)
    cur = local_conn.cursor()
    cur.close()
    return local_conn


def execute_q(conn, query):
    cur = conn.cursor()
    cur.execute(query)
    cur.close()
    conn.commit()


def query_q(conn, query):
    cur = conn.cursor()
    cur.execute(query)
    res = cur.fetchall()
    cur.close()
    conn.commit()
    return res
