#
# fileassets.py  -  Methods which allow to implement lazy dictionaries
#                   of objects residing in the file system.
#                   Most notably, it is useful to store SQL queries in
#                   dedicated SQL files.
#
# Copyright (C) 2016 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
'''Usually you access files in assets/xy/z.txt as "xy.z":

>>> fileassets['tests.a'] == u'Testing file.\\n'
True

We use file extensions only to provide the correct syntax highlighting
in files, in this example it's assets/tests/b.sql:

>>> (fileassets['tests.b'] ==
...  u"select 'h\\xe4tte, k\\xf6nnte, w\\xf6llte' as with_umlauts\\n")
True

Here's an internal test, which ensures that the file tests/a.txt is
not read twice but is correctly buffered in RAM. You do not need to
access the internal dictionary "data" directly.

>>> fileassets.data['tests.a']['payload'] == u'Testing file.\\n'
True

'''

# For directory listings
import os
# For checking time elapsed
import time
# For checking the modification time and
# for path name manipulations
import os.path

# For encoding
from .generic import to_ustring


class FileAssets:
    '''A file assets instance is a dictionary-like object, which will try
    to retrieve a key from a RAM dictionary, and if it is not present,
    will load data from the file system.

    Changes on the file system will be noted by the FileAssets
    instance, after a timeout, each object will be rechecked on disk.

    The keys of the objects are subpaths to the basepath, with periods
    instead of slashes and the file extension removed.
    '''

    def __init__(self, basepath=None, check_timeout=60):
        if basepath is None:
            basepath = os.path.join(
                os.path.dirname(__file__), 'assets')
        self.basepath = basepath
        self.check_timeout = check_timeout
        self.last_check = None
        self.data = {}
        self.unicode_exts = ['txt', 'py', 'sql', 'html', 'js', 'css']
        return

    def __getitem__(self, key):
        if key not in self:
            raise KeyError("Key not found in FileAssets")
        return self.data[key]['payload']

    def __setitem__(self, key, value):
        raise ValueError("FileAssets are read-only.")

    def __contains__(self, key):
        if key not in self.data or self.recheck(key):
            self.load(key)
        return key in self.data

    def get(self, key, default=None):
        '''Retrieve the given key.'''
        if key in self.data:
            return self.__getitem__(key)
        else:
            return default

    def find_file(self, key):
        '''Search a matching file.'''
        components = key.split('.')
        seekfull = os.path.join(self.basepath, *components)
        seekdir = os.path.dirname(seekfull)
        seekbase = os.path.basename(seekfull + '.')

        files = os.listdir(seekdir)
        matches = [match
                   for match in files
                   if match.startswith(seekbase) and not match.endswith('~')]
        if not matches:
            if key in self.data:
                del self.data[key]
            assert False, "File not found for "+str(key)

        assert len(matches) == 1, "Multiple matches found for "+str(key)

        fullname = os.path.join(seekdir, matches[0])
        return fullname

    def load(self, key):
        '''Seek for a file matching the key, and load it into the
        dictionary.'''

        filename = self.find_file(key)
        base, ext = os.path.splitext(filename)

        with open(filename, 'rb') as fh:
            payload = fh.read()

        if ext.strip('.') in self.unicode_exts:
            payload = to_ustring(payload)

        now = time.time()
        mtime = os.path.getmtime(filename)

        self.data[key] = {
            'ts': now,
            'mtime': mtime,
            'filename': filename,
            'payload': payload,
        }

    def recheck(self, key):
        '''Determine if it is time to recheck.'''
        now = time.time()
        entry = self.data.get(key, None)
        if not entry:
            return True
        if now - entry['ts'] < self.check_timeout:
            return False

        mtime = os.path.getmtime(entry['filename'])
        if mtime == entry['mtime']:
            return False
        return True


# Singleton instance
fileassets = FileAssets()


if __name__ == '__main__':
    import doctest
    doctest.testmod()
