#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# graphics.py  -  Pool of miscellaneous graphics utilities
#
# Copyright (C) 2015 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Requirements
# - Imagemagick (for 'convert')
# - netpbm (for 'pnm..')
# - ghostscript (for 'gs')
# - ploticus
# - inkscape
# - netpbm (pnmnorm, ppmtopgm, pgmtopbm)
# - libtiff-tools (ppm2tiff, tiffcp)
#
#

import os
import subprocess
import re
import tempfile
import colorsys
from .generic import to_string, to_bytes, safe_syscall


def call_ploticus(data, format='eps', dpi=100):
    '''ploticus.sf.net is a very smart plotting utility
    which can directly render PNG and EPS. PNG is a bit
    too pixelized for professional use.
    WARNING: be careful not to allow untrusted content to
    be passed as 'data' parameter, including anything that might
    be passed as parameter to a Zope object (since these can be set
    directly in the URL)
    '''
    valid_formats = ('eps', 'png', 'pdf', 'bmp', 'jpg', 'gif', 'ico')

    if not format:
        format = 'png'
    if format not in valid_formats:
        raise ValueError('invalid format!')

    tmpdir = tempfile.mkdtemp()

    opts = ['-eps', '-o', '%s/out.eps' % tmpdir]
    # if format in ('bmp', 'png', 'jpg', 'gif', 'ico'):
    #     opts += ' -png -o %s/out.png' % tmpdir
    # if format in ('pdf',): opts += ' -eps -o %s/out.eps' % tmpdir

    ploticus_cmd = ['/usr/bin/ploticus', '%s/in.txt' % tmpdir] + opts

    # ploticus needs latin1
    fd = open(tmpdir+'/in.txt', 'wb',)
    # if the input parameter is already "bytes", we expect it to have the
    # correct encoding (latin1).
    if not isinstance(data, bytes):
        data = data.encode('latin1', 'replace')
    fd.write(data)
    fd.flush()
    fd.close()

    ret, out = safe_syscall(ploticus_cmd)

    if format in ('bmp', 'png', 'jpg', 'gif', 'ico'):
        # Use method from eps_utils to convert EPS file
        fd = open(tmpdir+'/out.eps', 'rb')
        data = fd.read()
        fd.close()

        data = eps_to_png(data, res=dpi)

        fd = open(tmpdir+'/out.png', 'wb')
        fd.write(data)
        fd.close()

    data = convert_tmpfile(format, tmpdir)
    return data


def call_gnuplot(control, data):
    '''
    Gnuplot needs a control file and a data file to produce its output. Give
    both of these as arguments and fetch the output graph as return value. You
    may control the output file format as well, obviously.

    'control' should refer to the data file as %(data_file)s, so that
    formatting replacement can take place within this method. The same goes for
    %(output_file).

    WARNING: be careful not to allow untrusted content to be passed as
    'control' parameter, including anything that might be passed as parameter
    to a Zope object (since these can be set directly in the URL)
    '''

    tempdir = tempfile.mkdtemp()
    data_file = 'data'
    output_file = 'output'

    fh = open(tempdir+'/'+data_file, 'w')
    fh.write(data)
    fh.close()

    proc = subprocess.Popen(
        '/usr/bin/gnuplot', stdin=subprocess.PIPE, cwd=tempdir)
    proc.communicate(
        control % {'data_file': data_file, 'output_file': output_file})

    fh = open(tempdir+'/'+output_file, 'r')
    graph = fh.read()
    fh.close()

    # cleanup
    return graph


def call_inkscape(data, elem=None, format=None, dpi=None):
    '''
    '''
    valid_formats = ('eps', 'png', 'pdf', 'bmp', 'jpg', 'gif', 'ico')

    if not format:
        format = 'png'
    if format not in valid_formats:
        raise ValueError('invalid format!')

    tmpdir = tempfile.mkdtemp()

    # TODO: Use old options only if "-o" option leads to error code 1 with
    # ** (inkscape:3498604): WARNING **: 10:48:46.898: Invalid option -o

    old_opts = '-z'
    if format in ('bmp', 'png', 'jpg', 'gif', 'ico'):
        old_opts += ' -e %s/out.png' % tmpdir
        opts = '-o %s/out.png' % tmpdir
    if format in ('eps',):
        old_opts += ' -E %s/out.eps' % tmpdir
        opts = '-o %s/out.png' % tmpdir
    if format in ('pdf',):
        old_opts += ' -P %s/out.eps' % tmpdir
        opts = '-o %s/out.png' % tmpdir

    add_opts = ''
    if dpi:
        add_opts += ' -d %d' % int(dpi)
    if elem:
        add_opts += ' -i %s' % elem
    elif format in ('pdf',):
        add_opts += ' --export-area-page'

    inkscape_cmd = '/usr/bin/inkscape %s %s/in.svg' % (opts + add_opts, tmpdir)
    old_inkscape_cmd = '/usr/bin/inkscape %s %s/in.svg' % (
        old_opts + add_opts, tmpdir
    )

    with open(tmpdir+'/in.svg', 'w') as fd:
        fd.write(data)
        fd.flush()

    retcode, output = safe_syscall(inkscape_cmd.split(), raisemode=False)
    if retcode != 0:
        if 'Invalid option -o' in output:
            retcode, output = safe_syscall(
                old_inkscape_cmd.split(), raisemode=True
            )
        else:
            # Replicate the raise mode behavior
            quoted_cmd = ' '.join(
                ["'%s'" % a.replace("'", "\\'") for a in inkscape_cmd.split()]
            )
            raise AssertionError("return code %s [[%s]] on %s" %
                                 (retcode, output, quoted_cmd))

    data = convert_tmpfile(format, tmpdir)
    return data


def call_graphviz(source, program="dot", format="svg"):
    '''Call a graphviz program with the given source.

    Programs supported are: dot, neato, twopi, circo, fdp, sfdp,
    patchwork.

    Output formats supported are: ps, svg, png, gif, imap, cmapx

    For the explanation of the graph file language, see "man dot"

    A minimal graph might look like this:
    >>> graph = 'graph g0 { n0; n1; n2; n0 -- n1; n1 -- n2; }'

    This can be rendered as SVG:
    >>> (call_graphviz(graph)[0:60] ==
    ...  b'<?xml version="1.0" encoding="UTF-8" standalone="no"?>\\n<!DOC'
    ... )  # doctest: +SKIP
    True

    Or as PNG, for example:
    >>> (call_graphviz(graph, format='png')[0:18] ==
    ...  b'\\x89PNG\\r\\n\\x1a\\n\\x00\\x00\\x00\\rIHDR\\x00\\x00'
    ... )  # doctest: +SKIP
    True

    If an error occurs, as here (-> can only be used in digraph):
    >>> error = 'graph g0 { n0; n1; n2; n0 -> n1; n1 -> n2; }'

    Rendering it will result in a string starting with "Error:"
    >>> call_graphviz(error)[0:7] == b'Error: '  # doctest: +SKIP
    True

    '''
    legal_programs = [
        'dot', 'neato', 'twopi', 'circo',
        'fdp', 'sfdp', 'patchwork']
    legal_formats = ['ps', 'svg', 'png', 'gif', 'imap', 'cmapx']

    source = to_string(source)
    assert program in legal_programs, \
        "Illegal program. Use one of {}".format(legal_programs)
    assert format in legal_formats, \
        "Illegal format. Use one of {}".format(legal_formats)
    tmpdir = tempfile.mkdtemp()
    srcfile = tmpdir + '/in.gv'
    with open(srcfile, 'w') as fd:
        fd.write(source)
    retval, output = safe_syscall([program, '-T'+format, srcfile], text=False)
    return output


def call_plantuml(source, format="svg"):
    '''Call plantuml to convert simple UML into a picture.

    For the syntax of UML refer to http://plantuml.com:
    >>> graph = '@startuml\\n10\\n20\\n10 --> 20\\n@enduml\\n'

    >>> (call_plantuml(graph)[0:40] ==
    ...  b'<?xml version="1.0" encoding="UTF-8" sta')  # doctest: +SKIP
    True

    >>> (call_plantuml(graph, format="png")[0:18] ==
    ...  b'\\x89PNG\\r\\n\\x1a\\n\\x00\\x00\\x00\\rIHDR\\x00\\x00'
    ... )  # doctest: +SKIP
    True
    '''

    legal_formats = ['svg', 'png']
    source = to_string(source)
    assert format in legal_formats, \
        "Illegal format. Use one of {}".format(legal_formats)

    tmpdir = tempfile.mkdtemp()
    srcfile = tmpdir + '/graph.txt'
    with open(srcfile, 'w') as fd:
        fd.write(source)
    if format == 'svg':
        add_opts = ['-tsvg', ]
    else:
        add_opts = []

    retval, output = safe_syscall(['plantuml', ] + add_opts + [srcfile, ])
    outfile = tmpdir + '/graph.' + format
    with open(outfile, 'rb') as fd:
        data = fd.read()
    return data


def convert_tmpfile(format, tmpdir):
    '''Pass the eps or png file through appropriate filters to deliver the
    desired format.

    '''
    ret = ''
    if format == 'pdf':
        fd = os.popen('/usr/bin/epstopdf %s/out.eps' % tmpdir)
        ret = fd.read()
        fd.flush()
        fd.close()

    if format in ('jpg', 'bmp'):
        fd = os.popen('/usr/bin/convert %s/out.png %s/out.%s' %
                      (tmpdir, tmpdir, format))
        ret = fd.read()
        fd.flush()
        fd.close()

    if format in ('gif', 'ico'):
        fd = os.popen(
            '/usr/bin/pngtopnm -background "#FFFFFF" -mix %s/out.png | '
            '/usr/bin/ppmquant 256 | '
            '/usr/bin/ppmtogif -transparent "#FFFFFF"  > %s/out.gif' %
            (tmpdir, tmpdir))
        ret = fd.read()
        fd.flush()
        fd.close()

    if format == 'ico':
        fd = os.popen('/usr/bin/convert %s/out.gif  %s/out.ico' %
                      (tmpdir, tmpdir))
        ret = fd.read()
        fd.flush()
        fd.close()

    try:
        fd = open(tmpdir+'/out.'+format, 'rb')
    except IOError:
        if ret:
            raise ValueError(ret)
        else:
            raise

    data = fd.read()
    fd.close()
    return data


#
# Formerly eps_utils.py  -  EPS cleanup and translation utilities.
#


def translate_eps(eps):
    '''
    Find the bounding box in an EPS file and remove it
    and any (atend) candidates.
    Insert a translation that neutralizes the bbox offset.
    Recalculate and re-insert the new bbox.

    Return the new EPS code plus the bounding box.
    '''
    bbox = None
    lines = eps.split(b'\n')
    i = 0
    while True:
        if i >= len(lines):
            break
        line = lines[i]
        if line[:15] == b'%%BoundingBox: ':
            b = line.strip().split(b' ')[-4:]
            if len(b) == 4:
                bbox = list(map(int, b))
            lines.pop(i)
            continue
        i += 1

    if not bbox:
        return eps

    offset = (-bbox[0], -bbox[1])
    width = bbox[2] - bbox[0]
    height = bbox[3] - bbox[1]

    size = (width, height)

    bound = b'%%%%BoundingBox: 0 0 %d %d' % size
    trans = b' %d %d translate' % offset

    for i in range(len(lines)):
        if lines[i] and lines[i][:13] == b'%%EndComments':
            break
    lines.insert(i, bound)

    for i in range(len(lines)):
        if lines[i] and lines[i][0] != b"%":
            break
    lines.insert(i, trans)

    return (b'\n'.join(lines), size)


def recode_eps(eps):
    '''
    Find the fonts in the EPS file and see if it
    already uses ISOLatin1. If not, insert reencoding
    code.

    Return the enriched EPS.
    '''

    redef_font = b'''/%(fname)s findfont
dup length dict begin
{1 index /FID ne {def} {pop pop} ifelse} forall
/Encoding ISOLatin1Encoding def
currentdict
end
/%(fname)s exch definefont pop'''

    lines = eps.split(b'\n')
    corr = []

    fonts = {}
    has_iso = 0
    pat = re.compile(b'\\(([-A-Za-z0-9]*)\\) findfont')
    pat_iso = re.compile(b'ISOLatin1')
    for line in lines:
        match = pat_iso.search(line)
        if match:
            has_iso = 1

        match = pat.search(line)
        if match:
            for f in match.groups():
                fonts[f] = 1

    if has_iso:
        return eps

    for fname in fonts.keys():
        corr.append(redef_font % {'fname': fname})

    corr = b'\n'.join(corr)

    for i in range(len(lines)):
        if lines[i] and lines[i][0] != b"%":
            break
    lines.insert(i, corr)

    return b'\n'.join(lines)


def eps_to_png(eps, res=100, fonts=None):
    '''
    Use ghostscript to translate an EPS to PNG.
    '''
    if fonts:
        eps = recode_eps(eps)

    eps, dimen = translate_eps(eps)

    w, h = dimen
    w = int(float(w)*res/72.)
    h = int(float(h)*res/72.)
    size = '%dx%d' % (w, h)

    # Ugly patch for broken GS versions follows:
    eps += b'showpage\n'
    args = """/usr/bin/gs
            -q
            -dBATCH
            -dNOPAUSE
            -dTextAlphaBits=4
            -dGraphicsAlphaBits=4
            -sDEVICE=png16m
            -sOutputFile=-
            """.split() + ["-r{}".format(res), "-g{}".format(size), '-']
    ret, out = safe_syscall(args, stdin_data=to_bytes(eps), text=False)
    return out


def hsvtorgb(h, s, v):
    colorRGB = colorsys.hsv_to_rgb(h, s, v)
    return colorRGB


def pdf_to_tiff(data, dpi=300):
    '''Convert each page in a PDF into G4 encoded TIFF data and return the
    multipage TIFF.
    '''

    assert data, "No data"

    tmpdir = tempfile.mkdtemp()

    # First, convert to a series of bitmaps using Ghostscript
    cmd = ('cd %s; gs -q -dNOPAUSE -sDEVICE=ppmraw -r%d '
           '-dTextAlphaBits=4 -dGraphicsAlphaBits=4 '
           '-sOutputFile=%%03d.ppm - -c quit') % (tmpdir, dpi)
    fh = os.popen(cmd, 'w')
    fh.write(data)
    fh.close()

    # Treat each page separately
    pages = os.listdir(tmpdir)
    pages.sort()
    tiffs = ''
    for page in pages:
        if not page.endswith('.ppm'):
            continue
        base = page[:-4]
        # Convert each page to grey, then b/w, then tiff.
        cmd = ('cd %s; pnmnorm -bvalue 130 -wvalue 150 %s.ppm 2>/dev/null | '
               'ppmtopgm | pgmtopbm -threshold 2>/dev/null | '
               'ppm2tiff -R %d -c g4 %s.tiff') % (tmpdir, base, dpi, base)
        retcode = os.system(cmd)
        assert retcode == 0
        tiffs += ' %s.tiff' % base

    # Join the pages
    cmd = ('cd %s; tiffcp -c g4 %s out.tiff') % (tmpdir, tiffs)
    retcode = os.system(cmd)
    assert retcode == 0

    # Return the finished TIFF
    tiffdata = open(tmpdir + '/out.tiff', 'r').read()
    return tiffdata


def png_version(data, width=None, height=None):
    '''Generate a friendlier (png) version of an uploaded file using convert
    from ImageMagick.'''

    tempdir = tempfile.mkdtemp()
    infile = tempdir+'/infile'
    outfile = tempdir+'/outfile.png'
    args = ['/usr/bin/convert', ]
    if width:
        args += ['-resize', '%dx%d' % (width, height)]
    args += [infile, outfile]

    fh = open(infile, 'w')
    fh.write(data)
    fh.flush()
    fh.close()

    ret, out = safe_syscall(args)

    result = open(outfile, 'r').read() if ret == 0 else None
    return result
