# -*- coding: utf-8 -*-
#
# haproxy.py  -  Methods to communicate with haproxy via its admin socket.
#
# $Revision: 1.0 $
#
# Copyright (C) 2021 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

import socket
from .generic import to_bytes, to_string


def haproxy_send_cmd(cmd):
    '''Sends a command to the admin socket.
    Return the socket's output as a string.
    '''
    # connect to socket
    sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
    sock.connect(b'/var/run/haproxy/admin.sock')
    # send message
    sock.sendall(to_bytes(cmd)+b'\n')
    # read output
    readbuffer = sock.recv(4096)
    retmsg = readbuffer
    while len(readbuffer) == 4096:
        readbuffer = sock.recv(4096)
        retmsg += readbuffer

    return to_string(retmsg)


def haproxy_show_backend_servers(backend):
    '''Shows state of servers for a given backend
    Output is formatted like a table, header line begins with #,
    values are split with spaces.
    This output is returned as a more readable dictionary.
    '''
    # get information for backend
    server_res = haproxy_send_cmd(
        'show servers state {backend}'.format(backend=backend)
    )
    # cut off unnecessary lines at start and end
    lines = server_res.split('\n')[1:-2]
    # get keys of table, removing # at the start
    keys = lines[0].split(' ')[1:]
    # get values in each line
    values = [line.split(' ') for line in lines[1:]]
    # zip keys and values together
    servers = [dict(zip(keys, line)) for line in values]

    return servers


def haproxy_show_server_stats(backend, server=-1):
    '''Shows the detailed stats for a backend.
    Can optionally specify a server by its id to limit the result.
    '''
    server_stats = haproxy_send_cmd(
        'show stat {backend} 4 {server}'.format(backend=backend, server=server)
    )
    # cut off unnecessary lines at the end
    lines = server_stats.split('\n')[:-2]
    # get keys of table, removing # at the start
    keys = lines[0][2:].split(',')
    # get values in each line
    values = [line.split(',') for line in lines[1:]]
    # zip keys and values together
    stats = [dict(zip(keys, line)) for line in values]

    return stats


def haproxy_set_server_state(backend, server, state):
    '''Set a given server on a given backend into a given state.
    Valid states are:
        - 'ready':
            server is up and will accept connections
        - 'drain':
            server is up and will handle existing connections, but will
            not accept any new connections
        - 'maint':
            server is down
    Attempting to use any other state will result in an AssertionError.
    '''
    assert state in ['ready', 'drain', 'maint'], ('Target server state must be'
                                                  ' ready, drain or maint')

    haproxy_send_cmd(
        'set server {backend}/{server} state {state}'.format(
            backend=backend,
            server=server,
            state=state
        )
    )
