from icalendar import Calendar, Event, vCalAddress, vText
from datetime import datetime


def make_cal_entry(uuid, begin, end, attendees, description):
    """ Creates iCalendar entry

    :param uuid: uuid
    :type uuid: str
    :param begin: begin date in iso format
    :type begin: str
    :param end: end date in iso format
    :type end: str
    :param attendees: list of tuples (name, email)
    :type attendees: list[tuple]
    :param description: description
    :type description: str
    :return: calendar bytes-like object

    >>> make_cal_entry(
    ...     "824cdcab-fe48-4f72-a104-113011325f2b",
    ...     "2021-06-01 08:00", "2021-06-05 16:00",
    ...     [("name", "email@email.com"), ], "description"
    ... ) == (
    ...     b'BEGIN:VCALENDAR\\r\\nVERSION:1.0\\r\\nPRODID:perfact-ical\\r\\n'
    ...     b'UID:824cdcab-fe48-4f72-a104-113011325f2b\\r\\n'
    ...     b'BEGIN:VEVENT\\r\\nDTSTART;'
    ...     b'VALUE=DATE-TIME:20210601T080000\\r\\n'
    ...     b'DTEND;VALUE=DATE-TIME:20210605T160000\\r\\nATTENDEE;'
    ...     b'CN=name:MAILTO:email@email.com\\r\\n'
    ...     b'DESCRIPTION:description\\r\\nEND:VEVENT\\r\\n'
    ...     b'END:VCALENDAR\\r\\n'
    ... )
    True

    >>> make_cal_entry(
    ...     "b704457b-2b8c-45fe-bb35-dbeadd3533c8",
    ...     "2021-05-31", "2021-05-31",
    ...     [("new_name", "new@email.com"), ],  "description"
    ... ) == (
    ...     b'BEGIN:VCALENDAR\\r\\nVERSION:1.0\\r\\nPRODID:perfact-ical\\r\\n'
    ...     b'UID:b704457b-2b8c-45fe-bb35-dbeadd3533c8\\r\\n'
    ...     b'BEGIN:VEVENT\\r\\nDTSTART;'
    ...     b'VALUE=DATE-TIME:20210531T000000\\r\\n'
    ...     b'DTEND;VALUE=DATE-TIME:20210531T000000\\r\\nATTENDEE;'
    ...     b'CN=new_name:MAILTO:new@email.com\\r\\n'
    ...     b'DESCRIPTION:description\\r\\nEND:VEVENT\\r\\n'
    ...     b'END:VCALENDAR\\r\\n'
    ... )
    True

    """
    cal = Calendar()
    cal.add('uid', uuid)
    cal.add('prodid', "perfact-ical")
    cal.add('version', '1.0')

    event = Event()
    event.add('description', description)
    event.add('dtstart', datetime.fromisoformat(begin))
    event.add('dtend', datetime.fromisoformat(end))

    for name, email in attendees:
        attendee = vCalAddress("MAILTO:{}".format(email))
        attendee.params['cn'] = vText(name)
        event.add('attendee', attendee, encode=0)

    cal.add_component(event)

    return cal.to_ical()


def make_vcard(data):
    """ Creates vCard-formatted text for address book data.
    To be saved as a .vcf file.

    Input must be a dictionary of the following keys:
      lastname       - string, last name
      firstname      - string, first name
      salutation     - string, salutation
      acadtitle      - string, title
      lang           - string, language (following RFC 5646)
      locale         - string, locale (following RFC 5646)
      company        - string, name of the company
      department     - string, department / position in the company
      mail_addresses - array of strings, email addresses in order
                       of preference
      phone_numbers  - array of tuples, each two strings:
                       type of phone number & phone number

    Output follows RFC 6350 specificaton:
      BEGIN:VCARD
      VERSION:4.0
      REV:<timestamp>
      KIND:individual
      N:<last name>;<first name>;;<salutation & title>;
      FN:<formatted name>
      LANG:<lang>-<locale>
      ORG:<company>
      TITLE:<department>
      EMAIL;PREF=<order>:<email>
      [...]
      TEL;VALUE=uri;PREF=<order>;TYPE="<phone type>":tel:<phone number>
      [...]
      END:VCARD
    """

    last_name = data.get('lastname', '')
    first_name = data.get('firstname', '')
    salutation = data.get('salutation', '')
    acadtitle = data.get('acadtitle', '')

    # build prefix as array, since it contains optionals
    name_prefix = []
    if salutation:
        name_prefix.append(salutation)
    if acadtitle:
        name_prefix.append(acadtitle)

    # build display name with optionals
    display_name = []
    if acadtitle:
        display_name.append(acadtitle)
    if first_name:
        display_name.append(first_name)
    if last_name:
        display_name.append(last_name)

    vcard = [
        'BEGIN:VCARD',
        'VERSION:4.0',
        'REV:{}'.format(datetime.now().strftime("%Y%m%dT%H%M%S")),
        'KIND:individual',
        'N:{lastname};{firstname};;{prefix};'.format(
            lastname=last_name,
            firstname=first_name,
            prefix=' '.join(name_prefix),
        ),
        'FN:{}'.format(' '.join(display_name)),
        'LANG:{lang}-{locale}'.format(
            lang=data.get('lang', ''),
            locale=data.get('locale', ''),
        ),
        'ORG:{}'.format(data.get('company', '')),
        'TITLE:{}'.format(data.get('department', '')),
    ]
    # add mail addresses in order of preference
    for idx, mail in enumerate(data.get('mail_addresses', [])):
        vcard.append(
            'EMAIL;PREF={pref}:{adr}'.format(
                pref=idx+1,
                adr=mail,
            )
        )
    # add phone numbers in order of preference
    for idx, entry in enumerate(data.get('phone_numbers', [])):
        vcard.append(
            'TEL;VALUE=uri;PREF={pref};TYPE="{type}":{number}'.format(
                pref=idx+1,
                type=entry[0],
                number=entry[1],
            )
        )
    vcard.append('END:VCARD')
    return '\n'.join(vcard)
