#
# logbackup.py  -  Methods for dumping and restoring database tables.
#
# Copyright (C) 2018 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

import tempfile
import os

from .generic import safe_syscall

# Deployment on old systems may need an extra step:
# Edit /var/lib/zope2.13/instance/ema/Products/PerFactScriptModules/__init__.py
# and add a line: "add allow_module("perfact.logbackup")"

# destination where subfolders for ticket research are created
logs_save_path = '/var/ticket_logs/'
logbackup_path = '/opt/perfact/logbackup/'


def is_active():
    '''
    Check if root permissions for scripts in /opt/perfact/logbackup are
    enabled.
    '''
    return os.path.isfile('/etc/sudoers.d/logbackup')


def logs_overview():
    '''
    Return a list of all stored logs in $logs_save_path
    '''
    ls_tickets_cmd = ['ls', logs_save_path]
    ret, out = safe_syscall(
        ls_tickets_cmd,
        raisemode=False
    )
    if str(ret) != '0':
        return out
    folders = out.split()
    folder_list = []
    for folder in folders:
        path = logs_save_path + folder
        folder_item = {'ticket': path}
        ls_dates_cmd = ['ls', path]
        ret, out = safe_syscall(
            ls_dates_cmd,
            raisemode=False
        )
        folder_item['ret_code'] = ret
        if str(ret) == '0':
            folder_item['dates'] = out.split()
        else:
            folder_item['error'] = out
        folder_list.append(folder_item)
    return folder_list


def create_folder(incident_date, incident_ticket):
    '''
    Create a folder for storing logs: $logs_save_path/ticket_logs/
    $incident_ticket/$incident_date_$suffix
    '''
    log_folder_base = logs_save_path + \
        str(incident_ticket) + '/' + str(incident_date)

    success = False
    suffix = ''
    tries = 0
    while success is False:
        log_folder = log_folder_base + suffix
        logfolder_cmd = ['sudo', logbackup_path +
                         '/scripts/check_folder.sh', log_folder]
        ret, out = safe_syscall(
            logfolder_cmd,
            raisemode=False
        )
        if str(ret) != '0':
            # "ls $log_folder" fails, means the folder does not exist and can
            # be created
            success = True
        else:
            tries += 1
            suffix = '_' + str(tries)

    create_cmd = ['sudo', logbackup_path +
                  '/scripts/create_folder.sh', log_folder]
    ret, out = safe_syscall(
        create_cmd,
        raisemode=False
    )
    return ret, out, log_folder


def deactivate_permission():
    '''
    Deactivate all permissions that are needed to run scripts from
    /opt/perfact/logbackup
    with root permissions.
    '''
    deactivate_cmd = ['sudo', logbackup_path + '/scripts/deactivate']
    ret, out = safe_syscall(
        deactivate_cmd,
        raisemode=False
    )
    return ret, out


def archive_logs(incident_date, incident_ticket, log_formats):
    '''
    Find all log files that contain log lines for the incident_date.
    Create a folder for the incident_ticket and create an empty archive there.
    Copy all logfiles to this archive that contain information for the given
    incident_date.
    The log_format contains information regarding individual date formatting
    of each log.
    '''
    # create folder where logs can be copied to
    ret, out, log_folder = create_folder(incident_date, incident_ticket)
    if str(ret) != '0':
        deactivate_permission()
        return {'error': 'out', 'ret_code': ret}

    # create an empty tar archive as container for log files
    archive_name = log_folder + '/' + log_folder.split('/')[-1] + '.tar'
    create_cmd = ['sudo', logbackup_path +
                  '/scripts/create_archive.sh', archive_name]
    ret, out = safe_syscall(
        create_cmd,
        raisemode=False
    )
    if str(ret) != '0':
        return {'error': 'out', 'ret_code': ret}

    # check which logfiles contain the given date and copy them to the archive
    # we just created
    # make a list, which logs are missing (log not found/no log for the given
    # date)
    for item in log_formats:
        path = item['path']
        file_match = item['file_match']
        abspath_file_match = path + file_match
        regex = item['regex']
        # grep for date
        # sudo zgrep -l -e $regex $abspath_file_match
        grep_cmd = ['sudo', logbackup_path +
                    '/scripts/grep_for_date.sh', regex, abspath_file_match]
        ret, out = safe_syscall(
            grep_cmd,
            raisemode=False
        )
        if str(ret) != '0':
            item['error'] = out
            continue
        logfiles = out.split()
        for logfile in logfiles:
            add_cmd = [
                'sudo',
                logbackup_path + '/scripts/logbackup_add_to_tar.sh',
                archive_name,
                logfile,
            ]
            ret, out = safe_syscall(
                add_cmd,
                raisemode=True
            )

    # after storing logfiles to a tar archive, compress the archive
    gzip_cmd = ['sudo', logbackup_path +
                '/scripts/gzip_logfile.sh', archive_name]
    ret, out = safe_syscall(
        gzip_cmd,
        raisemode=True
    )

    # deactivate permissions that were granted for this method only
    deactivate_permission()

    # return: full path to the created folder, list of stored logs, list of
    # missing logs
    return {
        'results': log_formats,
        'log_folder': log_folder,
    }


def allowed_logfiles():
    '''
    !!! DEPRECATED - Legacy support only !!!
    List of logfiles that can be exported here.

    For security reasons each dictionary contains the key "key", which is used
    as a selector at the frontend and as a validator at the backend.
    Passing other filenames than those listed here becomes impossible this way.
    '''
    filelist = [
        {
            'key': 'syslog',
            'showname': 'Syslog',
            'abs_path': '/var/log/syslog'
        },
        {
            'key': 'perfactema',
            'showname': 'PerFactEMA',
            'abs_path': '/var/log/perfactema'
        },
        {
            'key': 'postgreslog',
            'showname': 'Postgres-Log',
            'abs_path': '/var/log/postgresql/postgresql-*-main.log'
        },
        {
            'key': 'accesslog',
            'showname': 'Apache-Access-Log',
            'abs_path': '/var/log/apache2/access.log'
        },
        {
            'key': 'apacheerrorlog',
            'showname': 'Apache-Error-Log',
            'abs_path': '/var/log/apache2/error.log'
        },
        {
            'key': 'eventlog',
            'showname': 'Zope-Event-Log',
            'abs_path': '/var/log/zope/ema*/event.log'
        },
        {
            'key': 'z2log',
            'showname': 'Zope-Z2-Log',
            'abs_path': '/var/log/zope/ema*/Z2.log'
        },
        {
            'key': 'cachelog',
            'showname': 'CacheManager-Log',
            'abs_path': '/var/log/cachemanager.log'
        },
        {
            'key': 'bookinglog',
            'showname': 'BookingManager-Log',
            'abs_path': '/var/log/bookingmanager.log'
        },
        {
            'key': 'measurelog',
            'showname': 'Measure-Log',
            'abs_path': '/var/log/measure.log'
        },
    ]
    return filelist


def logs_dump(keys, num_oldfiles=0, dryrun=False):
    '''
    !!! DEPRECATED - Legacy support only !!!
    This methods takes a list of keys that correspond to an entry in the list
    that is returned from the
    above defined method "allowed_logfiles".
    Taken those logfiles an archive (logs.tar.gz) is created and stored in a
    temporary folder.

    This functionality relies on the script
    /opt/perfact/logbackup/scripts/logbackup_add_to_tar.sh
    that is not itself part of the package

    Parameters:
    keys: Keys that correspond to filenames (from method "allowed_logfiles")
    num_oldfiles: extend to that older logfiles (i.e. log.1.gz ...)
    dryrun: only print out commands on STDOUT, no execution when the flag is
    set

    Output:
    Tuple of the path and the archive name itself (path, filename)
    '''
    # validate given keys and generate list of files to be exported
    valid_filelist = allowed_logfiles()
    filelist = []
    for key in keys:
        found = False
        for item in valid_filelist:
            if item.get('key') == key:
                found = True
                filelist.append(item['abs_path'])
                break
        if found is False:
            # when a key was passed from the frontend that does not exist
            # in the valid_filelist, this might be a security threat.
            # we only allow access to those files we have defined in the
            # valid_filelist (context/logfile_list)!
            raise AssertionError('Invalid key: ' + key)

    # Validate the parameter "num_oldfiles"
    # 1. it must be int
    try:
        num_oldfiles = int(num_oldfiles)
    except ValueError:
        raise ValueError('Parameter "num_oldfiles" must be int')
    # 2. it must be in a range of 0 - 52
    max_oldfilenum = 52
    if num_oldfiles not in range(max_oldfilenum+1):
        raise AssertionError(
            'Parameter "num_oldfiles" exceeds limit of ' + str(max_oldfilenum))

    # add older log files according to the parameter "num_oldfiles"
    old_files = []
    for idx in range(1, num_oldfiles+1):
        for logfile in filelist:
            # logfiles might be zipped, so generate two version (logfile.N,
            # logfile.N.gz)
            old_file = logfile + '.' + str(idx)
            old_file_gz = old_file + '.gz'
            old_files.append(old_file)
            old_files.append(old_file_gz)
    filelist += old_files

    # create destination folder and empty tar
    tmp_path = tempfile.mkdtemp()

    # create an empty tar archive as container for log files
    tar_filename = 'logs.tar'
    tar_file = tmp_path + '/' + tar_filename
    mktar_cmd = ['tar', '-rf', tar_file]
    if not dryrun:
        ret, out = safe_syscall(
            mktar_cmd,
            raisemode=False
        )

    # add all the files that were returned from the find command
    for logfile in filelist:
        # this must be added to /etc/sudoers for the user "zope"
        # it is also not a part of this package:
        add_cmd = [
            'sudo', '/opt/perfact/logbackup/scripts/logbackup_add_to_tar.sh',
            tar_file, logfile,
        ]
        if not dryrun:
            ret, out = safe_syscall(
                add_cmd,
                raisemode=False
            )

    # make a .tar.gz archive
    gzip_cmd = ['gzip', tar_file]
    if not dryrun:
        ret, out = safe_syscall(
            gzip_cmd,
            raisemode=False
        )
    archive_filename = tar_filename + '.gz'

    return tmp_path, archive_filename
