#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# network.py  -  Tools for networking, among others for Steute gateways
#
# $Revision: 1.0 $
#
# Copyright (C) 2015 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor,
# Boston, MA  02110-1301, USA.
#
import socket
import ipaddress
import codecs
from . import generic
import sys

PY2 = (sys.version_info.major == 2)
MINOR = sys.version_info.minor
# Shadow unicode with str for python3
if not PY2:
    unicode = str

# Add subnet_of and supernet_of methods to ipaddr.IPv4Network for python3 < 3.7
# These methods are included by default in python2 and python3 > 3.6
# The get_free_network function can stay unchanged by doing this
if not PY2 and MINOR < 7:
    def _is_subnet_of(a, b):
        try:
            # Always false if one is v4 and the other is v6.
            if a._version != b._version:
                raise TypeError("{} and {} are not of the same version".format(
                    a, b
                ))
            return (b.network_address <= a.network_address and
                    b.broadcast_address >= a.broadcast_address)
        except AttributeError:
            raise TypeError("Unable to test subnet containment"
                            " between {} and {}".format(a, b))

    def subnet_of(self, other):
        return self._is_subnet_of(self, other)

    def supernet_of(self, other):
        return self._is_subnet_of(other, self)

    ipaddress.IPv4Network._is_subnet_of = staticmethod(_is_subnet_of)
    ipaddress.IPv4Network.subnet_of = subnet_of
    ipaddress.IPv4Network.supernet_of = supernet_of


# Generic tools
def get_netmask(ipaddr):
    '''Extract the netmask from CIDR notation.

    >>> get_netmask('192.168.42.23') == u'255.255.255.255'
    True
    >>> get_netmask('172.16.0.0/12') == u'255.240.0.0'
    True
    '''
    net_obj = ipaddress.ip_network(unicode(ipaddr))
    return unicode(net_obj.netmask)


def ips_from_net(network, subnetmask, count=None, offset=0):
    '''
    return a list of ipv4 addresses from a network given by first ip and
    subnetmask.

    ::network    - address of an ipv4 network
    ::subnetmask - subnetmask of given ipv4 network
    ::count      - optional, how many addresses to return.
                   default is all possible
    ::offset     - optional, use negative offset to get ips from the back of
                   the network

    Basic test checking if function works as expected at all, compare against
    ipaddress
    >>> all_ips = ips_from_net('192.168.42.0', '255.255.255.0')
    >>> test_network = ipaddress.ip_network(u'192.168.42.0/255.255.255.0')
    >>> test_hosts = list(str(item) for item in test_network.hosts())
    >>> len(all_ips) == len(test_hosts)
    True

    Now check if ips are same, too
    >>> all_ips == test_hosts
    True

    Check count argument
    >>> all_ips = ips_from_net('192.168.42.0', '255.255.255.0', count=5)
    >>> len(all_ips) == 5
    True

    Count argument must not interfere with offset!
    >>> all_ips = ips_from_net('192.168.42.0', '255.255.255.0', count=5, \
    offset=5)
    >>> len(all_ips) == 5
    True

    Check offset argument - get 5 ips from net with positive offset 5
    >>> all_ips = ips_from_net('192.168.42.0', '255.255.255.0', count=5, \
    offset=5)
    >>> all_ips[-1] == u'192.168.42.10'
    True

    Check offset argument - get last 5 ips from net
    >>> all_ips = ips_from_net('192.168.42.0', '255.255.255.0', count=5, \
    offset=-1)
    >>> all_ips[-1] == u'192.168.42.254'
    True
    >>> len(all_ips)
    5
    '''

    # network constructor wants unicode!
    network = ipaddress.ip_network(
        generic.to_ustring(network) +
        generic.to_ustring('/') +
        generic.to_ustring(subnetmask)
    )

    hosts = list(network.hosts())

    if not count:
        numadrs = len(hosts)
    else:
        numadrs = count

    if not offset:
        res = hosts[:numadrs]
    else:
        if offset > 0:
            res = hosts[offset:offset + numadrs]
        else:
            # off-by-one for negative offsets...last ip must be accessible
            # reason: my_list[-1:0] yields no result
            if (offset + 1) != 0:
                res = hosts[(offset - numadrs) + 1:offset + 1]
            else:
                res = hosts[(offset - numadrs) + 1:]

    # we want plain strings as result
    res = [generic.to_ustring(item) for item in res]

    return res


def get_free_network(already_taken_networks, wanted_subnet, in_range):
    '''
    Get a available network with the given subnetmask.
    The network can't be taken and must be in the given subnet range

    :param already_taken_networks: Taken networks with CIDR as list of strings
    :param wanted_subnet: The subnet of the wanted network as int or string
    :param in_range: The network range in which the new one should be in as
    string
    :returns: A new network as string or nil string if no network could be
    found

    Basic tests following:

    >>> get_free_network(['10.1.0.0/24', '10.1.1.0/24'], 24, '10.1.0.0/16')
    '10.1.2.0/24'
    >>> get_free_network(['10.1.0.0/22', '10.1.4.0/22'], '24', '10.1.0.0/16')
    '10.1.8.0/24'
    >>> get_free_network(['10.1.0.0/22', '10.1.4.0/22'], '20', '10.1.0.0/16')
    '10.1.16.0/20'
    '''

    # Cast every network to unicode because the ipaddress packet want it so
    already_taken_networks = [unicode(net) for net in
                              already_taken_networks]

    # Network which will be (hopefully) selected later on
    selected = ''

    # Get a list of potentially working networks
    potential_networks = ipaddress.IPv4Network(
        unicode(in_range)).subnets(new_prefix=int(wanted_subnet)
                                   )

    # Iterate over every potential network
    for potential_network in potential_networks:
        # Iterate over the already taken networks
        for already_taken in already_taken_networks:
            already_taken = ipaddress.IPv4Network(already_taken)
            if potential_network == already_taken \
                    or already_taken.subnet_of(potential_network) \
                    or already_taken.supernet_of(potential_network):
                # If the potential network collides with one network which is
                # already taken, break out of the loop and try the next
                # potential network
                break
        else:
            # If we did not break out of the loop, we got a valid network
            # which did not collide with a already taken network
            selected = potential_network

        if selected:
            # If the potential network was valid we break out of the loop
            # to return the network
            break

    return str(selected)


def is_subnet_of(subnet, supernet):
    '''
    Check if the given subnet is actually a subnet of the given supernet

    :param subnet: Takes the subnet as string
    :param supernet: Takes the supernet as string
    :returns: True if the given subnet is actually
              a subnet of the given supernet else False

    Basic tests following:

    >>> is_subnet_of('10.222.4.0/24', '10.222.0.0/16')
    True
    >>> is_subnet_of('10.223.4.0/24', '10.222.0.0/16')
    False
    '''

    subnet = ipaddress.IPv4Network(unicode(subnet))
    supernet = ipaddress.IPv4Network(unicode(supernet))
    return subnet.subnet_of(supernet)


def is_supernet_of(supernet, subnet):
    '''
    Check if the given supernet is actually a supernet of the given subnet

    :param supernet: Takes the supernet as string
    :param subnet: Takes the subnet as string
    :returns: True if the given supernet is actually
              a supernet of the given subnet else False

    Basic tests following:

    >>> is_supernet_of('10.222.0.0/16', '10.222.4.0/24')
    True
    >>> is_supernet_of('10.222.0.0/16', '10.223.4.0/24')
    False
    '''
    return is_subnet_of(subnet, supernet)


class InterfacesParser(object):
    '''
    This is a parser for /etc/network/interfaces to read, understand,
    modify and save this file.

    Example Usage:
      # read from file
      ip = InterfacesParser(infile='/etc/network/interfaces')

      # read from string
      ip = InterfacesParser(content='auto eth0\niface eth0 inet static\n...')

      # get python data structure
      ip.interfaces
        { "eth0": {
            "auto": True,
            "method": "static",
            "options": {
                "address": "192.168.1.1",
                "netmask": "255.255.255.0",
                "gateway": "192.168.1.254",
                "dns-nameserver": "1.2.3.4",
            }
        }
        }

      # print /etc/network/interfaces like string from data
      ip.format()
        auto eth0
        iface eth0 inet static
            address 192.168.1.1
            netmask 255.255.255.0
            gateway 192.168.1.254
            dns-nameserver 1.2.3.4

      # save formatted string to file to create a /etc/network/interfaces file
      ip.save(filename='/tmp/interfaces')


    Original code by Cornelius Koelbel
    (https://github.com/privacyidea/networkparser)
    ----------------------------------
    The MIT License (MIT)

    Copyright (c) 2015 Cornelius Koelbel

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
    '''
    from pyparsing import Word, alphanums
    from pyparsing import Group, OneOrMore, ZeroOrMore, Optional
    from pyparsing import pythonStyleComment
    from pyparsing import LineStart, LineEnd
    from pyparsing import Regex, SkipTo

    interface = Word(alphanums + ':.')
    key = Word(alphanums + "-_")
    method = Regex(
        "loopback|manual|dhcp|static"
    )
    stanza = Regex(
        "(allow-)?auto|iface|mapping"
    )
    option_key = Regex(
        "bridge_\\w*|post-\\w*|up|down|pre-\\w*|address"
        "|network|netmask|gateway|broadcast|dns-\\w*|scope|"
        "pointtopoint|metric|hwaddress|mtu|hostname|"
        "leasehours|leasetime|vendor|client|bootfile|server"
        "|mode|endpoint|dstaddr|local|ttl|provider|unit"
        "|options|frame|netnum|media"
    )

    interface_block = Group(
        stanza + interface + Optional(
            Regex("inet") + method +
            Group(ZeroOrMore(Group(
                option_key + SkipTo(LineEnd())
            )))
        )
    )

    interface_file = OneOrMore(
        interface_block
    ).ignore(
        pythonStyleComment
    ).ignore(
        Regex("source.*")
    ).ignore(
        Regex('allow-hotplug.*')
    )

    def __init__(self,
                 infile="/etc/network/interfaces",
                 content=None):
        self.filename = None
        if content:
            self.content = content
        else:
            self.filename = infile
            self._read()

        self.interfaces = self.get_interfaces()

    def _read(self):
        """
        Reread the contents from the disk
        """
        f = codecs.open(self.filename, "r", "utf-8")
        self.content = f.read()
        f.close()

    def get(self):
        """
        return the grouped config
        """
        if self.filename:
            self._read()
        if not self.content.endswith('\n'):
            self.content += '\n'
        config = self.interface_file.parseString(self.content)
        return config

    def save(self, filename=None):
        if not filename and not self.filename:
            raise Exception("No filename specified")

        # The given filename overrules the own filename
        fname = filename or self.filename
        f = open(fname, "w")
        f.write(self.format())
        f.close()

    def format(self):
        """
        Format the single interfaces e.g. for writing to a file.

        {"eth0": {
            "auto": True,
            "method": "static",
            "options": {
                "address": "1.1.1.1",
                "netmask": "255.255.255.0",
            }
         }
        }
        results in

        auto eth0
        iface eth0 inet static
            address 1.1.1.1
            netmask 255.255.255.0

        :param interface: dictionary of interface
        :return: string
        """
        output = ""
        for iface, iconfig in sorted(list(self.interfaces.items())):
            if iconfig.get("auto"):
                output += "auto %s\n" % iface
            elif iconfig.get("allow-auto"):
                output += "allow-auto %s\n" % iface

            output += "iface %s inet %s\n" % (iface, iconfig.get("method",
                                                                 "manual"))
            # options
            for opt_key, opt_value in sorted(list(
                        iconfig.get("options", {}).items()
                    )):
                if not isinstance(opt_value, list):
                    output += "    %s %s\n" % (opt_key, opt_value)
                else:
                    # Option value is a list, add a line for each list item
                    output += '\n'.join(['    %s %s' % (opt_key, value)
                                         for value in opt_value]) + '\n'
            # add a new line
            output += "\n"
        return output

    def get_interfaces(self):
        """
        return the configuration by interfaces as a dictionary like

        { "eth0": {
            "auto": True,
            "method": "static",
            "options": {
                "address": "192.168.1.1",
                "netmask": "255.255.255.0",
                "gateway": "192.168.1.254",
                "dns-nameserver": "1.2.3.4",
            }
        }
        }

        :return: dict
        """
        interfaces = {}
        np = self.get()
        for idefinition in np:
            interface = idefinition[1]
            if interface not in interfaces:
                interfaces[interface] = {}
            # auto?
            if idefinition[0] == "auto":
                interfaces[interface]["auto"] = True
            elif idefinition[0] == "allow-auto":
                interfaces[interface]["allow-auto"] = True
            elif idefinition[0] == "iface":
                method = idefinition[3]
                interfaces[interface]["method"] = method
            # check for options
            if len(idefinition) == 5:
                options = {}
                for o in idefinition[4]:
                    opt = o[0]
                    value = o[1]
                    if opt not in options:
                        options[opt] = value

                    elif opt in options.keys():
                        # Option already present, turn into list and append
                        # new val
                        if not isinstance(options[opt], list):
                            options[opt] = [options[opt]] + [value]
                        # Already turned into list, simply append
                        else:
                            options[opt].append(value)

                interfaces[interface]["options"] = options
        return interfaces


def udpBroadcast(portnum, message, runs=10, bufsize=1024):
    '''Send a UPD broadcast with the given message to the given portnum
    and collect the responses'''
    # create the socket
    s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM, socket.IPPROTO_UDP)
    s.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, True)
    # sockets waits blocked for 5sec when receiving
    s.settimeout(5)

    s.sendto(message, ("<broadcast>", portnum))
    # receive loop collecting responses
    for i in range(runs):
        try:
            yield s.recvfrom(bufsize)
        except socket.timeout:
            break
    s.close()
