#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# pfcodechg.py  - Utilities for dbutils-zoperepo
#
#
# Copyright (C) 2017 Lars Bergmann <lars.bergmann@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

'''
**WIP-NOTES**
git --git-dir=/opt/perfact/dbutils-zoperepo/.git log
git --git-dir=/opt/perfact/dbutils-zoperepo/.git show <COMMIT>
x--format=format:"%H;%ad;%f
'''

# CUSTOM
from .server import get_pfsysteminfo
from .generic import safe_syscall, json_encode

# BUILTIN
import re
import subprocess

# MAILING
import smtplib
from email.mime.application import MIMEApplication
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate


def git_log(directory, log_arg=''):
    '''
    call git log, parse output and return result
    use optional log_arg to pass aditional argument to git log, e.g. "-1" to
    see only the last commit
    '''

    # see "man git log" for pretty handy format-shortcuts
    # TODO:check time format!
    call = 'git --git-dir={0}/.git log {1} --format=format:%H;%ai;%f'.format(
        directory, log_arg
    )
    retcode, output = safe_syscall(call, raisemode=True)

    # split into output lines and filter empty ones
    lines = [line for line in output.split('\n') if line]

    # pass iterator to zope?
    result = []
    for line in lines:
        commit_raw = line.split(';')
        commit = {
            'id': commit_raw[0],
            'date': commit_raw[1],
            'message': commit_raw[2]
        }
        result.append(commit)

    return result


def parse_diff(diff):
    '''
    extract meta information from raw diff
    '''

    diff_lines = diff.split('\n')

    # diff --git a/perfact/latex.py b/perfact/latex.py
    # get path without b/ --> relative path in repo
    path = diff_lines[0].split()[-1][2:]

    # TODO: also check for new file or deletion
    return path


def git_show(directory, commit):
    '''
    call git show for commit, parse output and return result
    '''

    # "format=" to omit commit header
    call = [
        'git',
        '--git-dir={}/.git'.format(directory),
        'show',
        commit,
        '--format=',
    ]
    retcode, output = safe_syscall(call)

    # someone had the same problem
    # https://stackoverflow.com/questions/10472224/
    # splitting-a-diff-file-using-regex-in-python
    diff_re_stmnt = re.compile(
        r'^(diff.*?)(?=^diff|\Z)',
        re.M | re.S
    )

    diffs = []
    for diff in re.findall(diff_re_stmnt, output):
        diff_d = {
            'diff_raw': diff.decode('utf-8', 'replace'),
            'path': parse_diff(diff).decode('utf-8'),
        }
        diffs.append(diff_d)
    return diffs


def git_init(directory, commit_email, commit_name):
    '''
    Initialize and configure a git repository in the given directory.
    '''

    call = 'git init {0}'.format(directory)
    retcode, output = safe_syscall(call, raisemode=True)

    call = 'git --git-dir={0}/.git config user.mail {1}'.format(
        directory,
        commit_email
    )
    retcode, output = safe_syscall(call, raisemode=True)

    call = 'git --git-dir={0}/.git config user.name {1}'.format(
        directory,
        commit_name
    )
    retcode, output = safe_syscall(call, raisemode=True)


def git_snapshot(directory, commit_message, subdir=None):
    '''
    stage all files in git repo of directory and commit
    returns True if commit was performed, False if not, e.g. if nothing to
    commit

    it's important to use --work-tree here to ensure git does not touch files
    outside the directory
    subdir: Optional. Pass a string containing the subpath to commit.
    '''
    call = [
        'git',
        '--git-dir={}/.git'.format(directory),
        '--work-tree={}'.format(directory),
        'add',
        '--all',
    ]
    if subdir:
        call.append(subdir)
    retcode, output = safe_syscall(call, raisemode=True)
    call = [
        'git',
        '--git-dir={}/.git'.format(directory),
        '--work-tree={}'.format(directory),
        'commit',
        '-m',
        commit_message,
    ]
    retcode, output = safe_syscall(call)

    # git returns error code if there's nothing to commit
    if retcode == 0:
        return True
    else:
        print(call)
        print(output)
        return False


def git_mail(directory, email_address):
    '''
    read last commit, parse diff and send it to given email address
    '''

    pfsystemid = get_pfsysteminfo()
    assert pfsystemid != 'unknown', 'Pfsystemid not found!'

    # log_arg = -1 means we want to see only the last commit
    commit = git_log(directory, log_arg='-1')[0]

    diffs = git_show(directory, commit['id'])
    diffs_json = json_encode(diffs)

    msg = MIMEMultipart()
    msg['From'] = 'codechanges@perfact.de'
    msg['To'] = email_address
    msg['Date'] = formatdate(localtime=True)
    msg['Subject'] = ';'.join([commit['id'], pfsystemid, commit['date']])

    msg.attach(MIMEText('auto commit mail'))

    json_filename = '{}.json'.format(commit['id'])
    json_attachment = MIMEApplication(diffs_json, Name=json_filename)
    json_attachment['Content-Disposition'] = (
        'attachment; filename="%s"' % json_filename)

    msg.attach(json_attachment)

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(
        'codechanges@perfact.de',
        [email_address, ],
        msg.as_string()
    )

    return


def git_mail_summary(directory, email_address):
    '''
    read last commit and send a summary to the given address.
    '''
    pfsystemid = get_pfsysteminfo('id')
    pfsystemname = get_pfsysteminfo('name')
    assert pfsystemname != 'unknown' and pfsystemid != 'unknown', \
        'PfSystemID not found!'

    status = subprocess.check_output(
        ['git', '-C', directory,
         'show', '--name-status', 'HEAD',
         ],
        universal_newlines=True,
    )

    msg = MIMEText(status, 'plain', 'utf-8')
    msg['Subject'] = 'Commit summary on %s (%s)' % (pfsystemname, pfsystemid)
    msg['To'] = email_address
    msg['From'] = 'codechanges@perfact.de'

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(
        msg['From'],
        [email_address, ],
        msg.as_string()
    )
    smtp.quit()
