#!/usr/bin/env python
# -*- coding: utf-8 -*-

import unittest

from .. import haproxy
from ..generic import to_string
from .mock_helper import ModulePatches


class TestHAProxy(unittest.TestCase):

    demo_return_servers = (
        b'1\n# be_id be_name srv_id srv_name srv_addr srv_op_state '
        b'srv_admin_state srv_uweight srv_iweight srv_time_since_last_change '
        b'srv_check_status srv_check_result srv_check_health srv_check_state '
        b'srv_agent_state bk_f_forced_id srv_f_forced_id srv_fqdn srv_port '
        b'srvrecord\n10 zope_instances 1 zope1 127.0.0.1 2 0 1 1 18675 15 3 4 '
        b'6 0 0 0 - 9081 -\n10 zope_instances 2 zope2 127.0.0.1 2 0 1 1 18709 '
        b'15 3 4 6 0 0 0 - 9082 -\n10 zope_instances 3 zope3 127.0.0.1 2 0 1 1'
        b' 18709 15 3 4 6 0 0 0 - 9083 -\n\n'
    )

    demo_return_stats = (
        b'# pxname,svname,qcur,qmax,scur,smax,slim,stot,bin,bout,dreq,dresp,'
        b'ereq,econ,eresp,wretr,wredis,status,weight,act,bck,chkfail,chkdown,'
        b'lastchg,downtime,qlimit,pid,iid,sid,throttle,lbtot,tracked,type,rate'
        b',rate_lim,rate_max,check_status,check_code,check_duration,hrsp_1xx,'
        b'hrsp_2xx,hrsp_3xx,hrsp_4xx,hrsp_5xx,hrsp_other,hanafail,req_rate,'
        b'req_rate_max,req_tot,cli_abrt,srv_abrt,comp_in,comp_out,comp_byp,'
        b'comp_rsp,lastsess,last_chk,last_agt,qtime,ctime,rtime,ttime,'
        b'agent_status,agent_code,agent_duration,check_desc,agent_desc,'
        b'check_rise,check_fall,check_health,agent_rise,agent_fall,'
        b'agent_health,addr,cookie,mode,algo,conn_rate,conn_rate_max,conn_tot,'
        b'intercepted,dcon,dses,wrew,connect,reuse,cache_lookups,cache_hits,'
        b'srv_icur,src_ilim,qtime_max,ctime_max,rtime_max,ttime_max,\n'
        b'zope_instances,zope1,0,0,0,6,10,525803,308814148,11734122435,,0,,0,0'
        b',7,4,UP,1,1,0,18,2,151283,0,,1,10,1,,525796,,2,2,,27,L7OK,200,24,0,'
        b'420269,105431,43,48,0,,,,,1,0,,,,,0,OK,,0,0,120,937,,,,Layer7 check '
        b'passed,,2,3,4,,,,127.0.0.1:9081,,http,,,,,,,,0,60801,465002,,,0,,'
        b'3004,39,52162,59937,\nzope_instances,zope2,0,0,0,6,10,532087,'
        b'312665697,11919778484,,0,,0,0,0,0,UP,1,1,0,11,0,177384,0,,1,10,2,,'
        b'532087,,2,2,,27,L7OK,200,4,0,426221,105777,49,40,0,,,,,1,0,,,,,0,OK,'
        b',0,0,95,821,,,,Layer7 check passed,,2,3,4,,,,127.0.0.1:9082,,http,,,'
        b',,,,,0,61157,470930,,,1,,3008,40,48194,60060,\nzope_instances,zope3,'
        b'0,0,1,6,10,532258,312972269,11853042659,,0,,0,1,12,7,UP,1,1,0,19,2,'
        b'151637,0,,1,10,3,,532246,,2,3,,29,L7OK,200,4,0,425800,106336,48,52,0'
        b',,,,,0,1,,,,,0,OK,,0,0,103,956,,,,Layer7 check passed,,2,3,4,,,,127.'
        b'0.0.1:9083,,http,,,,,,,,0,61035,471223,,,1,,0,33,46851,60070,\n\n'
    )

    # patch socket's recv method to return a dummy value
    @staticmethod
    def patchSocketOutput(mobj, output):
        socket_instance = mobj.socket.socket.return_value
        recv_method = socket_instance.recv

        def mock_recv(length):
            ret = output[mock_recv.index:mock_recv.index+length]
            mock_recv.index += length
            return ret
        mock_recv.index = 0

        recv_method.side_effect = mock_recv
        return recv_method

    # test the basic send command
    def test_haproxy_send_cmd(self):
        with ModulePatches(haproxy, ['socket']) as mobj:

            self.patchSocketOutput(mobj, self.demo_return_servers)

            # execute
            resp = haproxy.haproxy_send_cmd(
                'show servers state zope_instances')

            # assert correct output
            assert resp == to_string(self.demo_return_servers)

    # a long response
    def test_haproxy_long_response(self):
        with ModulePatches(haproxy, ['socket']) as mobj:

            self.patchSocketOutput(mobj, 'xxxx'*3000)

            # execute
            resp = haproxy.haproxy_send_cmd('dummy command')

            # assert correct output
            assert len(resp) == 12000

    # retrieve info on a backend
    def test_haproxy_show_backend_servers(self):
        with ModulePatches(haproxy, ['socket']) as mobj:

            self.patchSocketOutput(mobj, self.demo_return_servers)

            # execute
            res = haproxy.haproxy_show_backend_servers('zope_instances')

            # spotcheck the return values
            assert len(res) == 3
            assert res[0]['be_name'] == 'zope_instances'
            assert res[1]['srv_name'] == 'zope2'
            assert res[2]['srv_admin_state'] == '0'

    # retrieve specific stats on one or multiple servers
    def test_haproxy_show_server_stats(self):
        with ModulePatches(haproxy, ['socket']) as mobj:

            self.patchSocketOutput(mobj, self.demo_return_stats)

            # execute for all servers in the (mocked) backend
            res = haproxy.haproxy_show_server_stats('zope_instances')

            # spotcheck the return values
            assert len(res) == 3
            assert res[1]['svname'] == 'zope2'
            assert res[2]['scur'] == '1'
