#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# test_logging.py  -  test perfact.logging
#
# Copyright (C) 2018 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import unittest
import perfact.loggingtools
import logging

import argparse


class LoggingTC(unittest.TestCase):

    def getExampleParser(self):
        parser = argparse.ArgumentParser(description='An example')
        parser.add_argument(
                        '--port',
                        '-P',
                        type=int,
                        default=8200,
                        help='Port to bind to (defaults to 8200)')
        return parser

    def test_args(self):
        parser = self.getExampleParser()
        perfact.loggingtools.addArgs(parser, name='test')
        args = parser.parse_args(args=['-P', '400', '-l', 'console'])
        self.assertEqual(args.port, 400)
        self.assertEqual(args.log_to, 'console')
        self.assertEqual(args.log_file, '/var/log/test.log')
        self.assertEqual(args.debug, logging.INFO)

    def test_args_loglevel(self):
        parser = self.getExampleParser()
        perfact.loggingtools.addArgs(parser, name='test')
        args = parser.parse_args(args=['-P', '400', '-l', 'console', '-d'])
        self.assertEqual(args.debug, logging.DEBUG)
        args = parser.parse_args(
                        args=['-P', '400', '-l', 'console', '--debug'])
        self.assertEqual(args.debug, logging.DEBUG)

    def test_args_logfile(self):
        logfile_dest = '/var/log/logging/somethingelse.txt'
        parser = self.getExampleParser()
        perfact.loggingtools.addArgs(parser, name='test')
        args = parser.parse_args(
                    args=['-P', '400', '--log-to', 'file', '-f', logfile_dest])
        self.assertEqual(args.log_to, 'file')
        self.assertEqual(args.log_file, logfile_dest)
        args = parser.parse_args(
                args=['-P', '400', '-l', 'file', '--log-file', logfile_dest])
        self.assertEqual(args.log_to, 'file')
        self.assertEqual(args.log_file, logfile_dest)

    def test_create_logger_from_args(self):
        parser = self.getExampleParser()
        perfact.loggingtools.addArgs(parser, name='test')
        args = parser.parse_args(
                            args=['-P', '400', '--log-to', 'console', '-d'])
        logger = perfact.loggingtools.createLogger(args=args, name='test')
        self.assertIsInstance(logger, logging.Logger)
        self.assertEqual(logger.getEffectiveLevel(), logging.DEBUG)
        # has to check GE because every test gets the same logger
        self.assertGreaterEqual(len(logger.handlers), 1)

    def test_create_logger_from_object(self):
        args = perfact.loggingtools.LoggingArgs(
                                            log_to='journal',
                                            log_file='',
                                            log_level=logging.INFO)
        logger = perfact.loggingtools.createLogger(args=args, name='test')
        self.assertIsInstance(logger, logging.Logger)
        self.assertEqual(logger.getEffectiveLevel(), logging.INFO)
        # has to check GE because every test gets the same logger
        self.assertGreaterEqual(len(logger.handlers), 1)
