#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# test_xls.py  -  Test perfact.xls
#
# Copyright (C) 2022 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import unittest
from .. import fileassets
from .. import xls

# pyExcelerator xls writer format, pyexcel_write_xls
TEST_SHEET = [
    (
        u'Sheet1',
        {(0, 0): u'Foo', (1, 0): u'Bar', (2, 0): u'Ham', (3, 0): u'Eggs'}
    ),
]

TEST_TABLE = [[u'Foo'], [u'Bar'], [u'Ham'], [u'Eggs']]

TEST_DICTS = [{u'Foo': u'Bar'}, {u'Foo': u'Ham'}, {u'Foo': u'Eggs'}]

TEST_SHEET_EXTENDED = [
    {
        'sheetname': TEST_SHEET[0][0],
        'contents': TEST_SHEET[0][1],
    },
]

TEST_SHEET_EXTENDED_ALL_OPTIONS = [
    {
        'sheetname': TEST_SHEET[0][0],
        'contents': TEST_SHEET[0][1],
        'merged_cells': [{'from': (4, 0), 'to': (4, 4), 'label': 'Big Cell'}],
        'widths': {4: 1.5},
        'heights': {0: 0.45, 1: 0.9, 2: 1.35},
        'horz_page_breaks': [10, 20, 30],
        'fit_page': True,
        'freeze_row': 4,
        'freeze_column': 5,
    },
]

TEST_TEMPLATE_DATA = {
    'Sheet1': {
        'A5': u'Something',
    },
}

TEST_TEMPLATE_TABLE = [[u'Foo'], [u'Bar'], [u'Ham'], [u'Eggs'], [u'Something']]

max_content_length = 32767

TEST_SHEET_LONG_CONTENT = {
    (0, 0): u'x'*(max_content_length + 1),
    (1, 0): [
        (u'this is ', ''),
        (u'bold', 'bold on'),
        (u'x'*(max_content_length + 1), ''),
        (u'x', ''),
    ],
    (2, 0): True
}

TEST_SHEET_EXTENDED_LONG = [
    {
        'sheetname': TEST_SHEET[0][0],
        'contents': TEST_SHEET_LONG_CONTENT,
    },
]


class xls_tc(unittest.TestCase):

    def test_pyexcel_read_xls(self):
        testfile = fileassets.fileassets['tests.test_xls']
        parsed = xls.pyexcel_read_xls(testfile)
        self.assertEqual(parsed, TEST_SHEET, 'XLS read')

    def iter_test_helper(self, assetname, use_colmap=False):
        """Used by some tests below which have a lot in common. Set up callback
        that is called for each row in the given Excel file and check that it
        is correctly called."""
        testfile = fileassets.fileassets[assetname]
        lines = []

        def callback(rowidx, *args, **kw):
            lines.append((rowidx, kw['mycol'] if use_colmap else args[0]))

        xls.read_excel_iter(testfile, assetname, callback,
                            colmap={0: 'mycol'} if use_colmap else None)

        expect_dict = TEST_SHEET[0][1]
        expect = [
            (key[0], expect_dict[key])
            for key in sorted(expect_dict.keys())
        ]
        self.assertEqual(lines, expect,
                         'Excel read iter: {} != {}'.format(lines, expect))

    def test_pyexcel_read_xls_iter(self):
        self.iter_test_helper('tests.test_xls')

    def test_pyexcel_read_xlsx_iter(self):
        self.iter_test_helper('tests.test_xlsx')

    def test_pyexcel_read_xlsx_iter_colmap(self):
        self.iter_test_helper('tests.test_xlsx', use_colmap=True)

    def test_pyexcel_struct2table(self):
        testfile = fileassets.fileassets['tests.test_xls']
        parsed = xls.pyexcel_read_xls(testfile)
        table = xls.pyexcel_struct2table(parsed)
        self.assertEqual(table, TEST_TABLE, 'XLS struct2table')

    def test_pyexcel_table2struct(self):
        testfile = fileassets.fileassets['tests.test_xls']
        parsed = xls.pyexcel_read_xls(testfile)
        table = xls.pyexcel_struct2table(parsed)
        struct = xls.pyexcel_table2struct(table, sheetname='Sheet1')
        self.assertEqual(struct, TEST_SHEET, 'XLS table2struct')

    def test_pyexcel_table2_dict(self):
        testfile = fileassets.fileassets['tests.test_xls']
        struct = xls.pyexcel_read_xls(testfile)
        table = xls.pyexcel_struct2table(struct)
        dicts = xls.pyexcel_table2dicts(table)
        self.assertEqual(dicts, TEST_DICTS, 'XLS table2dict')

    def test_pyexcel_write_xls(self):
        testfile = fileassets.fileassets['tests.test_xls']
        orig = xls.pyexcel_read_xls(testfile)
        struct = xls.pyexcel_table2struct(TEST_TABLE, sheetname='Sheet1')
        testbytes = xls.pyexcel_write_xls(struct, mode='xls')
        newfile = xls.pyexcel_read_xls(testbytes)
        self.assertEqual(newfile, orig, 'XLS write_xls')

    def test_pyexcel_write_xlsx(self):
        testfile = fileassets.fileassets['tests.test_xls']
        orig = xls.pyexcel_read_xls(testfile)
        testbytes = xls.pyexcel_write_xls(orig, mode='xlsx')
        newfile = xls.read_workbook(testbytes)
        self.assertEqual(newfile, orig, 'XLS write_xlsx')

    def test_pyexcel_read_write_xlsx(self):
        testfile = fileassets.fileassets['tests.test_xlsx']
        orig = xls.read_workbook(testfile)
        testbytes = xls.pyexcel_write_xls(orig, mode='xlsx')
        newfile = xls.read_workbook(testbytes)
        self.assertEqual(newfile, orig, 'XLS read rite_xlsx')

    def test_pyexcel_write_extended(self):
        '''This call simply verifies that none of the passed arguments
        result in an error in the xlwt library.
        Unlike the other tests, no sophistated check of written data is
        done here, because styling, linebreaks and merged cells are not
        read from the file, so their success can not be verified.
        '''
        testbytes = xls.pyexcel_write_xls_extended(
            TEST_SHEET_EXTENDED_ALL_OPTIONS
        )
        self.assertIsNotNone(testbytes, 'XLS write_xls_extended')

    def test_pyexcel_write_extended_and_read(self):
        testfile = fileassets.fileassets['tests.test_xls']
        orig = xls.pyexcel_read_xls(testfile)
        testbytes = xls.pyexcel_write_xls_extended(TEST_SHEET_EXTENDED)
        newfile = xls.pyexcel_read_xls(testbytes)
        self.assertEqual(newfile, orig, 'XLS write_xls_extended_and_read')

    def test_write_workbook(self):
        write = xls.write_workbook(TEST_SHEET)
        assert write is not None
        assert isinstance(write, bytes)
        assert len(write) > 0

    def test_pyexcel_write_long_extended(self):
        '''Verify that too long strings will be cut. Strings are not allowed
        to be longer than 32767 characters.
        Check different content types:
            1. Content as a single string
            2. Content as a list of strings
        '''
        testbytes = xls.pyexcel_write_xls_extended(
            TEST_SHEET_EXTENDED_LONG
        )
        self.assertIsNotNone(testbytes, 'XLS write_xls_extended')

    '''
    # pyexcel_convert requires unoconv

    def test_data_from_template(self):
        testfile = fileassets.fileassets['tests.test_xlsx']
        testbytes = xls.pyexcel_book_from_template(
            testfile, TEST_TEMPLATE_DATA
        )
        xlsbytes = xls.pyexcel_convert(testbytes)
        struct = xls.pyexcel_read_xls(xlsbytes)
        table = xls.pyexcel_struct2table(struct)
        self.assertEqual(table, TEST_TEMPLATE_TABLE, 'XLS write_xls_template')
    '''
