#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# LDAP.py  -  Query a remote LDAP server for data.
#
# Copyright (C) 2018 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import ldap
import os


def ldap_query(uri, bases, manager_dn, manager_pw, cn,
               pw=None, filterstr=None, attrlist=None,
               timeout=3.0, nested_groups=False, cacertfile=None):
    '''Try to establish a connection to a LDAP server and search for
    given string in all given paths.
    Applies an optional filter and attribute list for narrowing down
    the results.
    Checks for validity of password if exactly 1 result is found and
    password is given.

    Returns a list which can be empty if no matches were found.
    Otherwise it contains tuples consisting of the distinguished name
    and a dictionary with all attributes.
    e.g. [('CN=test,OU=users,DC=example,DC=com',
           {'telephoneNumber': ['12345'],
            'mail': 'test@example.com'
           }
          )]

    If an error occured the first list item contains a tuple with
    'False' and a dictionary containing error details in 'msg' key.

    :uri
    URI containing protocol, host and port
    e.g. 'ldap://example.com:389'

    :bases
    List of paths to use for searching
    e.g. ['OU=users,DC=example,DC=com', 'OU=external,OU=other,DC=tld']

    :manager_dn
    Distinguished name for manager account
    e.g. 'CN=admin,OU=users,DC=example,DC=com'

    :manager_pw
    Password for manager account

    :cn
    Common name of an object, may be incomplete or including wildcards
    e.g. '*foo*' > objects containing 'foo'

    :filterstr
    This LDAP filter will be used for every search,
    default is to set 'sAMAccountName=$cn'.
    A LDAP filter can consist of multiple criterias e.g.:

    (|(cn=foo)(cn=bar))
      > every object where cn=foo OR cn=bar

    (!(cn=foo))
      > every object where cn!=foo

    (&(cn=foo)(sn=bar)
      > every object where cn=foo and sn=bar

    (&(|(cn=foo)(sn=bar))(objectCategory=person))
      > every object where (cn=foo OR sn=bar) AND objectCategory=person

    :attrlist
    List of LDAP attributes which should be returned as a result.
    e.g.  ['cn','telephoneNumber', 'mail']

    :pw
    If a password is given and exactly 1 result is found, then 'pw'
    will be checked for this object.

    :timeout
    A timeout in seconds (float) after which we fail if the server
    could not be reached in that time.

    :nested_groups
    Boolean to control additional query of all nested group memberships
    of this user. Usually only direct memberships ('memberOf') will be shown.
    The resultset will contain a special key 'nested_groups' with a list of all
    group names.
    Keep in mind that the groups you want to see must be contained in the given
    LDAP 'base'!

    :cacertfile
    Path to x509 certificate in the local filesystem containing a CA
    certificate to be used as verification for LDAPS (SSL) connections.
    The trusted certificates in /etc/ssl/certs will automatically be used!
    You only need to specify a file, if the certificate is not trusted by
    the system.
    '''
    conn = ldap.initialize(uri)
    conn.set_option(ldap.OPT_NETWORK_TIMEOUT, timeout)

    if cacertfile:
        assert os.path.exists(cacertfile), 'File %s does not exist!' \
            % cacertfile

        # certificates are small - 1 MB or more is suspicious!
        assert os.path.getsize(cacertfile) < 1024000, \
            'File %s too big (1MiB limit)!' % cacertfile

        # set a CA certificate file to be checked against for LDAPS
        conn.set_option(ldap.OPT_X_TLS_CACERTFILE, cacertfile)
        conn.set_option(ldap.OPT_X_TLS_NEWCTX, 0)

    try:
        conn.bind_s(manager_dn, manager_pw)
    except (ldap.SERVER_DOWN,
            ldap.INVALID_CREDENTIALS) as e:
        args = e.args[0]
        msg = args['desc']
        if 'info' in args:
            msg += '. ' + args['info']
        return [(False, {'msg': msg})]

    if filterstr is None:
        filterstr = '(sAMAccountName=%s)' % cn

    if attrlist is None:
        attrlist = [
            'cn',
            'sAMAccountName',
            'mail',
            'name',
            'displayName',
            'memberOf',
            'telephoneNumber',
            'userAccountControl',
        ]

    res = []
    for path in bases:
        curr = conn.search_s(
            base=path,
            scope=ldap.SCOPE_SUBTREE,
            filterstr=filterstr,
            attrlist=attrlist
        )
        if len(curr) > 0:
            if nested_groups:
                dn = curr[0][0]
                if not dn:
                    # in rare case the LDAP server returns an "empty" result
                    # e.g.
                    # [(None, ['ldaps://DomainDnsZones.example.com/' +
                    # 'DC=example,DC=com'])]
                    continue
                # Escape special characters as described in:
                # https://social.technet.microsoft.com/wiki/contents/articles/\
                # 5392.active-directory-ldap-syntax-filters.aspx#Special_Ch\
                # aracters
                # ToDo: Add support for language specific special characters
                # like Umlauts and so on
                # Maybe ldap.dn.escape_dn_chars can be used?
                # This needs to be tested!
                dn = dn.replace('(', '\28').replace(')', '\29')
                dn = dn.replace('*', '\2A').replace('\\', '\5C')
                # special LDAP OID for getting ALL memberships of given DN
                # see: https://docs.microsoft.com/de-de/windows/desktop/ADSI/\
                # search-filter-syntax
                # and: https://ldap.com/ldap-oid-reference-guide/
                group_filter = '(member:1.2.840.113556.1.4.1941:=' + dn + ')'
                groups = conn.search_s(
                    base=path,
                    scope=ldap.SCOPE_SUBTREE,
                    filterstr=group_filter,
                    attrlist=attrlist,
                )
                curr[0][1]['nested_groups'] = groups
                # for debugging we add the created filter to the resultset
                curr[0][1]['nested_groups_filter'] = group_filter
            res = res+curr

    # check credentials for single result (e.g. User)
    if len(res) and pw:
        # Validate user by trying to bind
        try:
            dn = res[0][0]
            conn.bind_s(dn, pw)
            check = ['passed']
        except ldap.INVALID_CREDENTIALS:
            check = ['failed']
        res[0][1]['password_check'] = check

    conn.unbind()
    return res


def test_all():
    uri = 'ldap://example.com:389'
    bases = ['OU=Users,DC=example,DC=com']
    manager_dn = 'CN=admin,OU=Users,DC=example,DC=com'
    manager_pw = 'geheim'
    cn = 'testbenutzer'
    pw = 'secret'

    result = ldap_query(
        uri=uri,
        bases=bases,
        manager_dn=manager_dn,
        manager_pw=manager_pw,
        cn=cn,
        pw=pw,
        filterstr=None,
        attrlist=None
    )
    print(str(result))

    # print all key/value pairs of resultset
    for ident, d in result:
        print(ident)
        for key, values in d.items():
            print(" %s" % key)
            for val in values:
                print("  %s" % val)
        print("")
