#!/usr/bin/python
#
# it3000.py  -  Utilities to control an IT3000 balance.
#
# Copyright (C) 2014 Jan Jockusch <jan.jockusch@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import socket
import time
import sys
import syslog
from perfact.generic import to_bytes, to_string


class IT3000:
    '''IT3000 class: Implements connecting to a balance and sending commands.
    '''

    def __init__(self, soc, debug=0):
        self.soc = soc
        self.debug = debug
        return

    def dbg(self, message, level=1):
        '''If the debug level is high enough, this prints a formatted message.
        '''
        if self.debug >= level:
            syslog.syslog('IT3000 - %s, %s' % (self.modem_addr, message))
        return

    def send(self, command, value=''):
        '''Send <command> to balance and return a dictionary.
        '''
        cmds = ('RN', 'RM', 'TM', 'TA')
        d = {'error': '99'}
        if command not in cmds:
            return d

        self.soc.send(to_bytes('<'+command+value+'>'))
        ret = b''
        while 1:
            c = self.soc.recv(1)
            if not c:
                break
            if c == b'\n':
                continue
            if c == b'\r':
                break
            ret += c
        ret = to_string(ret)

        d['fullreturnval'] = ret
        if not (len(ret) > 1 and ret[0] == '<' and ret[-1] == '>'):
            return d

        d['error'] = ret[1:3]
        d['data'] = ret[3:-1]
        if d['error'] != '00':
            return d

        # Parse return data for some commands
        if command in ('RN', 'RM'):
            d['status'] = ret[3:5]
            d['date'] = ret[5:13]
            d['time'] = ret[13:18]
            d['ident'] = ret[18:22].strip()
            d['balance_id'] = ret[22:23]
            d['gross'] = ret[23:31].strip()
            d['tara'] = ret[31:39].strip()
            d['net'] = ret[39:47].strip()
            d['unit'] = ret[47:49].strip()
            d['taracode'] = ret[49:51]
            d['range'] = ret[51:52]
            d['terminal_id'] = ret[52:55]
            d['crc16'] = ret[55:63]
        return d


if __name__ == '__main__':
    '''If called directly (e.g. via ssh for remote readouts) pass
    command line options as command and value.
    '''
    start = time.time()
    reset = 0
    debug = 0
    hostname = ''
    port = 0

    if len(sys.argv) < 4:
        print("Missing argument. Usage: it3000.py host port command [ value ]")
        sys.exit()

    me = sys.argv.pop(0)
    hostname = sys.argv.pop(0)
    port = int(sys.argv.pop(0))
    command = sys.argv.pop(0)
    try:
        value = sys.argv.pop(0)
    except IndexError:
        value = ''

    for token in sys.argv:
        if token == 'debug':
            debug += 1

    socket.setdefaulttimeout(1)
    soc = socket.socket()
    soc.connect((hostname, port))

    b = IT3000(soc, debug=debug)
    d = b.send(command, value)
    # return pickled dictionary
    print(str(d))
