#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# checktarget.py  -  Utilities for connection checking of a target host
#
# Copyright (C) 2016 Thorsten Südbrock <thorsten.suedbrock@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import socket
import subprocess


def checkTCPTarget(target_ip, target_port, timeout=4, source_ip=None):
    """Try to open a socket on a given ip and port

    >>> checkTCPTarget('127.0.0.1', 22)  # doctest: +SKIP
    (0, 'Socket open')

    >>> checkTCPTarget('', 65432)  # doctest: +SKIP
    (2, 'Socket closed, Connection refused')

    >>> checkTCPTarget('127.0.0.1', 22,
    ...     source_ip='127.0.0.2')  # doctest: +SKIP
    (0, 'Socket open')
    """
    d = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    d.settimeout(timeout)
    if source_ip:
        d.bind((source_ip, 0))
    try:
        d.connect((target_ip, target_port))
        d.close()
        return (0, 'Socket open')
    except socket.timeout:
        return (1, 'Socket timeout')
    except socket.error as detail:
        return (2, 'Socket closed, '+str(detail.args[1]))


def checkICMPTarget(target_ip, source_ip=None):
    """Sends an icmp echo request to the target ip and expects a echo reply
    If reply is received, output 0 (errorlevel of 'ping'), otherwise output
    the corresponding error code

    >>> checkICMPTarget('127.0.0.1')  # doctest: +SKIP
    0

    >>> checkICMPTarget('169.254.99.99')  # doctest: +SKIP
    1

    >>> checkICMPTarget('127.0.0.1', '127.0.0.2')  # doctest: +SKIP
    0
    """
    cmd = ['/bin/ping', '-c1', '-n', '-q', '-W1', target_ip]
    if source_ip:
        cmd = ['/bin/ping', '-c1', '-n', '-q', '-W1', '-I',
               source_ip, target_ip]
    ret = subprocess.call(cmd)
    return (ret)


def checkIPAddress(ip_address=''):
    """Reverse resolution of the given ip address to name

    >>> checkIPAddress('127.0.0.1')  # doctest: +SKIP
    'localhost'

    >>> checkIPAddress('169.254.99.99')  # doctest: +SKIP
    '**unknown**'
    """
    if not ip_address:
        return ''
    try:
        hinfo = socket.gethostbyaddr(ip_address)
    except socket.herror:
        return '**unknown**'
    return hinfo[0]


def checkFQDN(fqdn=''):
    """Returns the first IPv4 address of the given FQDN (hostname)

    >>> checkFQDN('localhost')  # doctest: +SKIP
    '127.0.0.1'

    >>> checkFQDN('nonexistent.example.com')  # doctest: +SKIP
    ''
    """
    if not fqdn:
        return ''
    try:
        result = socket.getaddrinfo(fqdn, None, socket.AF_INET)
        if result:
            ip_address = result[0][4][0]
    except socket.gaierror:
        return ''
    return ip_address


def checkFQDNex(fqdn=''):
    """Returns all IP addresses of the given FQDN (hostname) as a list

    >>> checkFQDNex('localhost')  # doctest: +SKIP
    ['127.0.0.1', '::1']

    >>> checkFQDNex('nonexistent.example.com')  # doctest: +SKIP
    []
    """
    if not fqdn:
        return ''
    try:
        ip_addresses = []
        results = socket.getaddrinfo(fqdn, None)
        if results:
            for ip in results:
                if ip[4][0] not in ip_addresses:
                    ip_addresses.append(ip[4][0])
    except socket.gaierror:
        return []
    ip_addresses.sort()
    return ip_addresses


def checkVNCViewerListenTarget(target_ip, target_port, timeout=4,
                               msg=b'RFB 003.008\n'):
    """Establishes a TCP-Connection with a VNC-Viewer in listening mode.
    Send the protocol version to the VNC-Viewer. If it gets echoed back to us
    everything is OK.
    If not we do not know if the other end is really there or can understand
    us and we exit

    >>> checkVNCViewerListenTarget('169.254.99.99', 65432,
    ...     timeout=1)  # doctest: +SKIP
    (1, 'Socket timeout')

    >>> checkVNCViewerListenTarget('127.0.0.1', 65432,
    ...     timeout=1)  # doctest: +SKIP
    (2, 'Socket closed, Connection refused')

    >>> checkVNCViewerListenTarget('127.0.0.1', 22,
    ...     timeout=1)  # doctest: +SKIP
    (3, 'Could not reach VNC-Viewer')
    """
    d = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    d.settimeout(timeout)
    try:
        d.connect((target_ip, target_port))
        d.send(msg)
        data = d.recv(1024)
        d.close()
        if data == msg:
            return (0, 'VNC-Viewer attached')
        else:
            return (3, 'Could not reach VNC-Viewer')
    except socket.timeout:
        return (1, 'Socket timeout')
    except socket.error as detail:
        return (2, 'Socket closed, '+str(detail.args[1]))
