#!/usr/bin/python
# coding: utf8
#
# ODataPseudoConnection.py  -  a pseudo db-connection for OData webservices
#                               used in dbconnector.py
#
# $Revision: 1.10 $
#
# Copyright (C) 2017 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# $Id: dbconnector.py,v 2.10 2013/09/10 13:29:47 perfact Exp perfact $

import netrc                # auth from file .netrc
import perfact.webservice
import requests


class ODataPseudoConnection(object):
    """Wrapper for webservice calls.

    Webservice calls are wrapped in functions similar to that of database
    connections.
    """

    def __init__(self, connector, dbtype=''):
        """Initilize a ODataPseudoConnection object."""
        self.connector = connector
        connector_parts = connector.split()
        self.dbtype = dbtype
        self.session = requests
        self.endpoint = connector_parts[0]
        self.auth = None
        self.token_endpoint = None

        if len(connector_parts) > 1:
            self.token_endpoint = connector_parts[1]

        if dbtype in ['odata_token', 'odata_token_patch', 'odata_noauth']:
            self.auth = None
        elif dbtype == 'odata':
            host = self.connector

            host = perfact.webservice.removeCredentialsFromUrl(host)
            if host.startswith('https'):
                host = host.replace('https://', '').split('/')[0]
            elif host.startswith('http'):
                host = host.replace('http://', '').split('/')[0]

            # get auth from file .netrc-odata, make sure this file exists and
            # contains the login informations for the host.
            odatalogin = netrc.netrc('/home/perfact/.netrc-odata')
            user, account, password = odatalogin.authenticators(host)
            self.credentials = {}
            self.credentials['username'], self.credentials['password'] = \
                user, password
            self.auth = (user, password)

    def connect(self):
        """Create a new session."""
        # logout when need to
        self.disconnect()
        # create a new session
        self.session = requests.Session()

    def disconnect(self):
        """Delete session."""
        self.session = None

    def close(self):
        """Delete session by using disconnect."""
        self.disconnect()

    def read(self, query, add_headers=None, session=requests, method='json',
             **kwargs):
        """Read data from the odata endpoint in the way method discrips."""
        if add_headers is None:
            add_headers = {}

        headers = {'Content-Type': 'application/json'}
        # Augment and quote the headers
        headers.update(add_headers)

        if not self.connected:
            self.connect()

        params = {'url': '%s/%s' % (self.endpoint, query),
                  'headers': headers,
                  'auth': self.auth,
                  }

        # set information for authentificatioin method
        if self.token_endpoint:
            params['token_url'] = self.token_endpoint
        if ('authtype' in kwargs):
            params['authtype'] = kwargs['authtype']

        methods = {
            'json': perfact.webservice.getJSON,
            'text': perfact.webservice.getText
        }

        return methods[method](**params)

    def write(self, data, add_headers=None, session=requests, method='post',
              **kwargs):
        """Write data to the odata endpoint in the way method discrips.

        input:
        - data: the data that should be writen to the endpoint.
        - add_headers: further headers that should be added to the request.
        optional input:
        session: a session that shuld be used for the request can be set.
                 the request object is used if none is given.
        method: request has diefferent methode how data can be written to an
                endpoint. the methods 'post' and 'patch' are implemented.
                If no methode is given 'post' is used.
        """
        if add_headers is None:
            add_headers = {}

        headers = {'Content-Type': 'application/json'}
        # Augment and quote the headers
        headers.update(add_headers)

        if not self.connected:
            self.connect()

        params = {'url': self.endpoint,
                  'headers': headers,
                  'data': data,
                  'auth': self.auth,
                  'verify': False,
                  }

        # set information for authentificatio nmethod
        if self.token_endpoint:
            params['token_url'] = self.token_endpoint
        if ('authtype' in kwargs):
            params['authtype'] = kwargs['authtype']

        methodes = {
            'post': perfact.webservice.postWS,
            'patch': perfact.webservice.patchWS
        }

        return methodes[method](**params)

    def __del__(self):
        """Delete the session by calling the disconnect function."""
        self.disconnect()

    @property
    def connected(self):
        """Give the connection status."""
        if self.session:
            return True
        return False
