#
# dbconn.py  -  Utilities for connecting with the DB-Utils database
#
# Copyright (C) 2016 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
'''
This module supports the following operations:

- Execute a simple statement:

  >>> dbconn.execute('select 6*7')  # doctest: +SKIP
  >>> dbconn.tuples()  # doctest: +SKIP
  [(42,)]

- Perform secure variable replacement in query templates:

  >>> dbconn.execute('select %s, %s, %s',
  ...                ['asdf', 15, ' \\'; drop table \\ntesting;'],
  ...               )  # doctest: +SKIP

  >>> (dbconn.query_compiled ==
  ...  b"select 'asdf', 15, ' ''; drop table \\ntesting;'")  # doctest: +SKIP
  True

  >>> dbconn.tuples()  # doctest: +SKIP
  [('asdf', 15, " '; drop table \\ntesting;")]


- Perform schema changes and insertions:

  >>> dbconn.execute('create temp table testing (a bigint)')  # doctest: +SKIP
  >>> dbconn.executemany(
  ...     'insert into testing values (%s)',
  ...     [
  ...      (123,),
  ...      (124,),
  ...      (125,),
  ...     ])  # doctest: +SKIP

- Access data via .tuples(), .dictionaries() or .result():

  >>> dbconn.execute('select * from testing')  # doctest: +SKIP
  >>> (dbconn.tuples() ==
  ...  [(123,), (124,), (125,)])  # doctest: +SKIP
  True
  >>> (dbconn.dictionaries() ==
  ...  [{'a': 123}, {'a': 124}, {'a': 125}])  # doctest: +SKIP
  True
  >>> dbconn.result()[1].a == 124  # doctest: +SKIP
  True

- Pass parameters as keyword arguments:

  >>> dbconn.execute('select * from testing where a = %(value)s',
  ...                value=123)  # doctest: +SKIP
  >>> dbconn.dictionaries() == [{'a': 123}]  # doctest: +SKIP
  True

- Clean handling of "None" (is passed as NULL):

  >>> dbconn.execute('select %(value)s is null', value=None)  # doctest: +SKIP
  >>> dbconn.query_compiled == b'select NULL is null'  # doctest: +SKIP
  True

  >>> dbconn.tuples()  # doctest: +SKIP
  [(True,)]

- Handling of binary data (needs esc_binary()):

  >>> from .sql import esc_binary  # doctest: +SKIP
  >>> dbconn.execute('select E%(value)s::bytea as binary',
  ...                value=esc_binary('ABCD\\x01\\x7f\\x02\\x00')
  ...               )  # doctest: +SKIP
  >>> (dbconn.query_compiled ==
  ...  b"select E'ABCD\\\\\\\\001\\\\\\\\177\\\\\\\\002\\\\\\\\000'"
  ...  b"::bytea as binary")  # doctest: +SKIP
  True

  >>> (bytes(dbconn.tuples()[0][0]) ==
  ...  b'ABCD\\x01\\x7f\\x02\\x00')  # doctest: +SKIP
  True

- Transaction handling (you may commit and rollback):

  >>> dbconn.rollback()  # doctest: +SKIP

- Wrapping of dictionaries to enable access as attributes:

  >>> a = DBConnResult({'a': 1234, 'b': 5678})
  >>> hasattr(a, 'b')
  True
  >>> a.a
  1234
  >>> a.b
  5678
  >>> del a['b']
  >>> hasattr(a, 'b')
  False

'''

import psycopg2
# For waiting on notifies
import select
# For escaping binary material


class DBConnResult(object):
    '''This class wraps around a dictionary, retaining all dictionary
    features, but adding class member access to the keys.'''

    def __init__(self, d):
        self.d = d

    def __getattr__(self, key):
        try:
            return getattr(self.d, key)
        except AttributeError:
            pass

        try:
            return self.d[key]
        except KeyError as e:
            raise AttributeError(e)

    def __delitem__(self, key):
        del self.d[key]


class DBConn:

    '''Class wrapper for database related services.'''

    def __init__(self,
                 dbconn_string='dbname=perfactema user=zope'):
        self.dbconn_string = dbconn_string
        self.dbconn = None
        return

    def connect(self):
        '''Connect to the DB-Utils database.
        '''
        self.dbconn = psycopg2.connect(self.dbconn_string)
        self.cursor = self.dbconn.cursor()
        return self.dbconn

    def disconnect(self):
        '''Disconnect from database.
        '''
        if (self.dbconn):
            self.dbconn.close()
            self.dbconn = None
            self.cursor = None
        return

    def commit(self):
        return self.dbconn.commit()

    def rollback(self):
        return self.dbconn.rollback()

    def execute(self, query, args=None, auto_fetch=True, **kw):
        '''Wrapper for cursor execute command. Reconnects if necessary.
        Performs an automatic fetchall() if auto_fetch is set.
        '''
        # Keyword arguments take precedence over non-keywords
        if kw:
            args = kw

        self.query = query

        if not self.dbconn:
            self.connect()

        try:
            self.cursor.execute(query, args)
        except Exception:
            self.connect()
            self.cursor.execute(query, args)
        if auto_fetch:
            self.fetchall()

        if args:
            self.query_compiled = self.cursor.mogrify(query, args)
        else:
            self.query_compiled = self.query

        return

    def executemany(self, query, args=None):
        '''Wrapper for cursor executemany command. Reconnects if necessary.
        '''
        self.query = query
        self.query_compiled = None

        if not self.dbconn:
            self.connect()

        try:
            self.cursor.executemany(query, args)
        except Exception:
            self.connect()
            self.cursor.executemany(query, args)
        return

    def fetchall(self):
        '''Wrapper for "fetchall" which also stores the result and caches the
        column names.

        This allows the repeated use of "tuples()" and "dictionaries()".

        '''
        if not self.cursor.description or self.cursor.rowcount == -1:
            # There's no result to fetch here
            self.result_data, self.names = [], []
        else:
            self.result_data = self.cursor.fetchall()
            self.names = [column.name
                          for column in self.cursor.description]
        return self.result_data

    def tuples(self):
        '''Similar API to ZSQL Methods. Returns the exact same list of tuples
        which fetchall() generates.
        '''
        return self.result_data

    def dictionaries(self):
        '''Similar API to ZSQL Methods. Returns a list of dictionaries indexed
        by column names.
        '''
        return [dict(zip(self.names, row))
                for row in self.result_data]

    def result(self):
        '''Similar API to ZSQL Methods. Returns a list of DBConnResult objects
        which allow accessing columns as attributes.
        '''
        return [DBConnResult(dict(zip(self.names, row)))
                for row in self.result_data]

    def setlisten(self, notifier):
        '''Bring the database connector into listening mode.
        '''
        self.execute('listen %s' % notifier)
        self.commit()

    def fetchnotifies(self, notifier):
        '''Fetch the number of notifies. This resets notifications for the
        next run.
        '''
        # Send a dummy query.
        self.execute('select 1')
        self.commit()
        notifiers = 0
        # Retrieve notifications, filtering down to the ones meant for us.
        for i in range(len(self.dbconn.notifies)-1, -1, -1):
            if self.dbconn.notifies[i].channel == notifier:
                notifiers += 1
                self.dbconn.notifies.pop(i)
        return notifiers

    def getnotifyhandler(self, notifier):
        '''Prepare the database for listen mode and return a file handler.'''
        self.setlisten(notifier)
        # Check notifiers before waiting
        notifiers = self.fetchnotifies(notifier)
        if notifiers:
            return notifiers
        fd = self.cursor.connection.fileno()
        return fd

    def waitnotifies(self, notifier, timeout=120):
        '''Put database in listen mode and wait for notifications.
        '''
        fd = self.getnotifyhandler(notifier)
        # Wait.
        readable, writable, exceptions = select.select([fd], [], [], timeout)
        if not readable:
            return 0
        return self.fetchnotifies(notifier)


# Singleton instance
dbconn = DBConn()
