import string
# For generating input for HTTPResponse
from io import BytesIO
# For hexadecimal encoding
import codecs
import six
# For simple route parsing
import re

# For response parsing
if six.PY2:
    from httplib import HTTPResponse
else:
    from http.client import HTTPResponse


def parse_http_response(response):
    '''Dissect a response into a dictionary consisting of "status", "headers"
    and "body". The headers are also stored in a dictionary, all keys are in
    lower case.

    Example:
    >>> resp = (
    ...     b'HTTP/1.1 200 OK\\nContent-Length: 6\\n'
    ...     b'Pragma: no-cache\\nContent-Type: text/plain; charset=utf-8\\n'
    ...     b'\\nTEST.\\n')
    >>> (parse_http_response(resp) ==
    ...  {'status': 200, 'headers': {'content-length': '6',
    ...   'pragma': 'no-cache', 'content-type': 'text/plain; charset=utf-8'},
    ...   'body': b'TEST.\\n'})
    True

    Zero bytes in the response are passed as they are.
    >>> resp = (
    ...     b'HTTP/1.1 200 OK\\nContent-Length: 2\\n'
    ...     b'Pragma: no-cache\\nContent-Type: text/plain; charset=utf-8\\n'
    ...     b'\\n\\x00\\x00')
    >>> (parse_http_response(resp) ==
    ...  {'status': 200, 'headers': {'content-length': '2',
    ...   'pragma': 'no-cache', 'content-type': 'text/plain; charset=utf-8'},
    ...   'body': b'\\x00\\x00'})
    True

    '''
    class FakeSocket():
        def __init__(self, response):
            self._file = BytesIO(response)

        def makefile(self, *args, **kwargs):
            return self._file
    source = FakeSocket(response)
    resp = HTTPResponse(source)
    resp.begin()
    body = resp.read()
    headers = {key.lower(): value for key, value in resp.getheaders()}
    status = resp.status
    return_value = {
        'status': status,
        'headers': headers,
        'body': body,
    }
    return return_value


def format_http_body(body):
    '''Shorten the output from the given response. The input body
    must be bytes. The output is a string.

    A short text will be passed as-is:
    >>> format_http_body(b'ABCDEF\\n') == u'ABCDEF\\n'
    True

    Unprintable bytes lead to a hex dump (if the output is short):
    >>> format_http_body(b'\\x00\\xff\\x00\\x80') == u'00ff0080\\n'
    True

    Many lines of output are shorted to just the first and the last lines:
    >>> body = 20 * b"Hello world!\\n"
    >>> (format_http_body(body) ==
    ...  u'Hello world!\\nHello world!\\n...\\nHello world!\\nHello world!\\n')
    True

    Long lines are shortened to just the first and the last characters:
    >>> (format_http_body(200 * b"A" + b"\\n") == u'AAAAAAAAAAAAAAAAAAAAAAAA'
    ...  'AAAAAAAAAAAAAA...AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA\\n')
    True

    Long unprintable output leads to a shortened hex dump with only the first
    and last bytes:
    >>> (format_http_body(2000 * b'\\xaa\\x55') == u'aa55aa55aa55aa55aa55aa55'
    ...  'aa55aa55aa55aa...55aa55aa55aa55aa55aa55aa55aa55aa55aa55\\n')
    True

    '''
    printable_set = set(string.printable.encode('ascii'))
    if set(body).issubset(printable_set):
        # Assume utf-8 encoding (would be safer to extract from
        # content-type)
        body = body.decode('utf-8').rstrip('\n')
        # Response is probably a text. Shorten if necessary
        if len(body) > 80:  # Magic number of maximum characters
            lines = body.split('\n')
            if len(lines) > 4:  # Magic number of maximum lines
                lines = lines[:2] + ['...', ] + lines[-2:]
            for index, line in enumerate(lines):
                if len(line) > 80:  # Magic number of max chars/line
                    lines[index] = line[:38] + '...' + line[-38:]
            body = '\n'.join(lines)
    else:
        # Response is probably binary. Shorten also.
        if len(body) > 40:  # Magic number of maximum bytes
            body = (codecs.encode(body[:19], 'hex') + b'...' +
                    codecs.encode(body[-19:], 'hex')).decode('ascii')
        else:
            body = codecs.encode(body, 'hex').decode('ascii')
    return body + '\n'


def map_routing(subpath, routes, fallback=None):
    '''Based on the given subpath and routes returns the first match.
    The regex groups are parsed and returned in an additional params
    dictionary. Due to the nature of python regex-matching the routes
    need to be provided from "specific" to less specific as an empty
    route '/' will match everything.

    subpath: list of str containing the whole subpath
    routes: list of dictionaries which contains a 'spec' and any
        arbitrary amount of other keys.

    returns a dictionary containing the parsed 'params' and the
        dictionary of the found route, as provided in 'routes'
    '''
    params = {}
    mapped_route = fallback

    for route in routes:
        if not route['spec'].startswith('/'):
            raise AssertionError(
                'Invalid route {spec} specified, needs to start with /'.format(
                    spec=route['spec']
                )
            )
        path = '/' + '/'.join(subpath)

        spec_regex = re.compile(route['spec'])
        re_match = re.match(spec_regex, path)

        if re_match:
            params.update(dict(re_match.groupdict()))
            mapped_route = route
            break

    if not mapped_route:
        raise NotImplementedError('No route matched')

    return {
        'route': mapped_route,
        'params': params
    }
