#
# latex.py  -  Methods to provide PDF documents including barcodes.
#
# Copyright (C) 2004 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import os
import tempfile
from . import generic
import subprocess
from .graphics import call_inkscape
from .generic import to_string, to_ustring, to_bytes
from .generic import get_type, cleanup_string
from .generic import base64decode
import string
import six

if not six.PY2:
    unicode = str
    buffer = memoryview


def barcode_pdflatex(source, context=None):
    'Alias for backward compatibility'
    return extended_pdflatex(source, context)


def generate_incfiles(source, tempdir, context=None):
    '''Parse a TeX source for special barcode and
    image tags. Barcode tags start with %%BARCODE_RAW,
    %%BARCODE_I25, %%BARCODE_EAN or just %%BARCODE.
    After this, just write the input to the 'barcode'
    program.

    Path tags start with %%PATH <path>. Path tags
    require that you pass a context to search in.
    The path is interpreted relative to the context.

    URL tags start with %%URL <url>. URL tags are
    retrieved by using curl. Craft the URL so that
    it works on the server!

    DATA tags start with %%DATA image/png;base64,<data>.
    Only this pattern is allowed. This will lead to
    writing out the PNG data into a file.

    All included images are stored in a temporary
    directory.

    '''
    # source to bytes when python2 or unicode when python3
    if isinstance(source, six.binary_type):
        source = to_string(source)

    miss_path = '/usr/share/texmf/tex/latex/pfstyles/abbildung-fehlt.pdf'

    count = 0
    lines = source.split('\n')
    ret = []
    while 1:
        if not lines:
            break
        line = lines.pop(0)
        if line[:9] == '%%BARCODE':
            if line[9:13] == '_RAW':
                code = line[14:]
                enc = '128raw'
            elif line[9:13] == '_I25':
                code = line[14:]
                enc = 'i25'
            elif line[9:13] == '_EAN':
                code = line[14:]
                enc = 'ean'
            else:
                code = line[10:]
                enc = '128'

            line = lines.pop(0)
            # Make barcode
            count += 1
            fname = 'inc%(count)04d.%(ext)s' % locals()
            code = code.replace('$', '\\$')
            assert False, "%% BARCODE not supported in this version"
            ret.append(fname + '.pdf' + line)
            continue

        if line.startswith('%%PATH '):
            path = line[7:].strip()
            if not context:
                raise ValueError('%%PATH used without context!')

            data, ctype = None, None
            try:
                # So far, this only works for images and files.
                obj = context.restrictedTraverse(path)
                data = obj.data
                ctype = obj.getContentType()
            except Exception:
                pass
            if data is None:
                try:
                    # Python Scripts could be in the way, which need
                    # traverse_subpath.  This situation is handled by a
                    # wrapper script in Zope.
                    ctype, data = context.layout_traverse(path)
                except Exception:
                    pass
            if data is None:
                # Last resort. use a replacement image
                miss = open(miss_path, 'rb')
                data = miss.read()
                miss.close()
                ctype = 'application/pdf'

            ext = ctype.lower().split('/')[-1]
            if ext not in ('png', 'pdf', 'jpeg'):
                raise ValueError('unsupported content type: '+ctype)

            line = lines.pop(0)
            count += 1
            fname = 'inc%(count)04d.%(ext)s' % locals()
            fh = open(os.path.join(tempdir, fname), 'wb')
            if isinstance(data, bytes):
                fh.write(data)
            elif isinstance(data, str):
                fh.write(to_bytes(data))
            elif isinstance(data, (buffer, memoryview)):
                start = 0
                while start < len(data):
                    fh.write(data[start:start+4096])
                    start += 4096
            else:
                # Handle linked pdata objects:
                while data is not None:
                    fh.write(to_bytes(data.data))
                    data = data.next
            fh.close()
            ret.append(fname+line)
            continue

        if line.startswith('%%URL '):
            ext = None
            valid_extensions = ('png', 'pdf', 'jpeg')

            url = line[6:].strip()
            url = cleanup_string(url, valid_chars=(
                string.ascii_letters + string.digits
                + ':/-_.,;#%!$()[]{}+*?&='))
            # Default extension from URL
            ext = url.rsplit('.', 1)[-1]
            if ext not in ('png', 'pdf', 'jpeg'):
                ext = None

            # Do not use -i if a file is dowloaded because
            # the header will corrupt the file.
            curl_includeheader = '-i'
            if url.startswith('file://'):
                curl_includeheader = ''

            proc = subprocess.Popen(
                ['/usr/bin/curl', '-s', '-m', '45',
                 curl_includeheader, '-L', url],
                stdout=subprocess.PIPE,
                universal_newlines=False,
            )
            fh = proc.stdout
            # Handle files which need conversion
            convert_from = None
            ctype = None
            if url.startswith('file://'):
                # Guess the extension from the URL
                guess_ext = url.rsplit('.', 1)[-1]
                if guess_ext in valid_extensions:
                    ext = guess_ext
            else:
                while True:
                    line = fh.readline().strip()
                    if line.lower().startswith(b'content-type: '):
                        ctype = to_string(line.lower()[14:])
                        ext = ctype.split('/')[-1]

                        if ext in ('svg+xml',):
                            # Handle some of the unsupported types by
                            # converting into supported ones.
                            convert_from = ext
                            ext = 'pdf'
                            # TODO: Can this be dynamically switched to "png"?

                        if ext not in valid_extensions:
                            raise ValueError(
                                "Unsupported content type: {}".format(ctype)
                            )

                    # Break on blank line (headers are over), but only if
                    # we have found a content type. This skips over header
                    # blocks from curl redirects.
                    if ctype and not line:
                        break

            data = fh.read()
            assert proc.wait() == 0, "Error calling curl"
            if not (data and ext):
                raise ValueError('error fetching URL: '+(url, len(data), ext))

            # Convert file if necessary
            if convert_from:
                if convert_from == 'svg+xml':
                    # Use other external method to perform the conversion
                    new_data = call_inkscape(data=data, format=ext)
                    if new_data:
                        data = new_data
                    else:
                        raise ValueError('conversion problem with inkscape!')

            line = lines.pop(0)
            count += 1
            fname = 'inc%(count)04d.%(ext)s' % locals()
            fh = open(os.path.join(tempdir, fname), 'wb')
            fh.write(data)
            fh.close()
            ret.append(fname+line)
            continue

        if line.startswith('%%DATA '):
            pattern = '%%DATA image/png;base64,'
            if line.startswith(pattern):
                try:
                    data = base64decode(line[len(pattern):])
                except Exception:
                    line = lines.pop(0)
                    ret.append(miss_path+line)
                    continue

                line = lines.pop(0)
                ext = 'png'
                count += 1
                fname = 'inc%(count)04d.%(ext)s' % locals()
                fh = open(os.path.join(tempdir, fname), 'wb')
                fh.write(data)
                fh.close()
                ret.append(fname+line)
                continue

        ret.append('\n' + line)
    ret = ''.join(ret)[1:]
    return ret


# Aliases for extended_tex_call:
def extended_xelatex(source, context=None, min_runs=1,
                     show_log=False, error_cb=None):
    return extended_texcall(source, context, tex_variant='xelatex',
                            min_runs=min_runs, show_log=show_log,
                            error_cb=error_cb)


def extended_pdflatex(source, context=None, min_runs=1,
                      show_log=False, error_cb=None):
    return extended_texcall(source, context, tex_variant='pdflatex',
                            min_runs=min_runs, show_log=show_log,
                            error_cb=error_cb)


def extended_texcall(source, context=None, tex_variant='pdflatex', min_runs=1,
                     show_log=False, error_cb=None):
    '''Generate include files in a temporary directory.
    Then, run pdflatex as long as it takes.
    '''
    # Validate
    assert tex_variant in ('pdftex', 'pdflatex', 'xetex', 'xelatex'), \
        "Invalid value given for tex_variant"

    # We don't support pdflatex anymore
    # Haven't worked before but now we officially say so
    if six.PY3 and isinstance(source, bytes):
        raise NotImplementedError("pdflatex not supported in python3")

    tempdir = tempfile.mkdtemp()
    ret = generate_incfiles(source, tempdir, context=context)
    if isinstance(ret, six.text_type):
        ret = to_bytes(ret)

    # Write parsed TeX source into the temporary directory
    texname = 'document.tex'
    fh = open(tempdir + '/' + texname, 'wb')
    fh.write(ret)
    fh.flush()
    fh.close()

    # Run pdfLaTeX as often as it takes.
    pat = 'Label(s) may have changed. Rerun'
    runs = 0
    timeout = 300
    while 1:
        runs += 1
        proc = subprocess.Popen(
            [
                '/usr/bin/timeout',
                str(timeout),
                tex_variant,
                '-interaction', 'errorstopmode',
                texname,
            ],
            cwd=tempdir,
            stdout=subprocess.PIPE,
            universal_newlines=True,
        )
        [ret, err] = proc.communicate()
        if proc.returncode != 0 or not os.path.isfile(tempdir+'/document.pdf'):
            lines = [
                line for line in ret.split('\n')
                if line.startswith('! LaTeX Error')
            ]
            if error_cb is not None:
                error_cb(ret=ret, err=err,
                         returncode=proc.returncode)
            raise AssertionError(
                lines[0] if len(lines) else 'LaTeX error'
            )
        if (ret.find(pat) == -1) and (runs > min_runs):
            break

    # Read PDF result.
    pdf = open(tempdir + '/document.pdf', 'rb').read()

    # If the option show_log is passed, we change the return value!
    if show_log:
        log = open(tempdir + '/document.log', 'r').read()
        return (pdf, log)

    # Return the PDF content
    return pdf


def make_pdfa(data):
    '''Use ghostscript to turn a PDF into archivable PDF-A'''
    tempdir = tempfile.mkdtemp()

    fh = open(tempdir + '/in.pdf', 'wb')
    fh.write(data)
    fh.flush()
    fh.close()

    cmd = ('cd %(tempdir)s ; gs -dPDFA -dBATCH -dNOPAUSE -dUseCIEColor '
           '-sProcessColorModel=DeviceCMYK -sDEVICE=pdfwrite '
           '-sPDFACompatibilityPolicy=1 -sOutputFile=outfile '
           'in.pdf' % locals())
    ret = os.popen(cmd, 'rb').read()
    # ret may contain error information
    data = open(tempdir + '/out.pdf', 'rb').read()
    return data


def tex_quote(value, ustrings=False):
    '''
    Quote TeX specials.
    The parameter `ustrings` forces the output to be converted to unicode
    (Python 2 only - in Python 3, it is always returned as unicode/text)
    '''
    if not value:
        return ''
    value = to_ustring(value)

    value = value.replace('\\', '\\textbackslash ')
    value = value.replace('{', '\\{')
    value = value.replace('}', '\\}')
    value = value.replace('&', '\\&')
    value = value.replace('#', '\\#')
    value = value.replace('$', '\\$')
    value = value.replace('%', '\\%')

    # Specialties for the german package
    value = value.replace('"', '{\\dq}')

    value = value.replace('_', '\\textunderscore{}')
    value = value.replace('^', '\\^{}')
    value = value.replace('~', '\\~{}')
    # value = value.replace('\xc2\xad', '') # soft hyphen replacement
    value = value.replace('\\textbackslash ', '\\textbackslash{}')

    if ustrings:
        value = to_ustring(value)
    else:
        value = to_string(value)
    return value


def esc_tex(value, ustrings=False):
    '''
    Escape strings so they can be used in TeX.
    The parameter `ustrings` forces the output to be converted to unicode
    (Python 2 only - in Python 3, it is always returned as unicode/text)
    '''
    if not value:
        return ''
    value = value.replace('\\', '\\textbackslash ')
    value = value.replace('{', '\\{')
    value = value.replace('}', '\\}')
    value = value.replace('"', "''")
    value = value.replace('#', '\\#')
    value = value.replace('&', '\\&')
    value = value.replace('$', '\\$')
    value = value.replace('%', '\\%')
    value = value.replace('_', '\\textunderscore{}')
    value = value.replace('^', '\\^{}')
    value = value.replace('~', '\\~{}')
    value = value.replace('\\textbackslash ', '\\textbackslash{}')

    # Remove substitute control character
    value = value.replace('\x1a', '')

    if ustrings:
        value = to_ustring(value)
    else:
        value = to_string(value)
    return value


def tex_cleanup(value, ustrings=False, strip=False, escape=True):
    '''
    Create a copy of a nested structure, cleaning up all values for usage in
    TeX.

    The parameter `ustrings` forces the output to be converted to unicode
    (Python 2 only - in Python 3, it is always returned as unicode/text)

    `strip` causes the contained strings to be stripped.

    `escape` can be used to pass a custom escape function. By default, esc_tex
    is used. By passing `False`, escaping can be turned off explicitly.

    >>> tex_cleanup(None)
    ''
    >>> tex_cleanup(1)
    '1'
    >>> tex_cleanup(1.5)
    '1.5'
    >>> tex_cleanup(['test', 'aoeu', 5])
    ['test', 'aoeu', '5']
    >>> tex_cleanup('test{}')
    'test\\\\{\\\\}'
    >>> tex_cleanup('test', ustrings=True) == u'test'
    True
    >>> tex_cleanup({1: 'aoeu', 2: 'test_'}) == (
    ...     {1: 'aoeu', 2: 'test\\\\textunderscore{}'})
    True
    >>> tex_cleanup(u'test', ustrings=True) == u'test'
    True
    >>> tex_cleanup(u'test', ustrings=False) == 'test'
    True
    >>> tex_cleanup('test\x1a') == 'test'
    True
    '''

    def cleanup(obj):
        'Recursively clean up'

        if isinstance(obj, (list, tuple)):
            return [cleanup(item) for item in obj]

        if isinstance(obj, dict):
            return {
                key: cleanup(value)
                for key, value in obj.items()
            }

        if obj is None:
            return ''

        obj = to_string(obj)
        if strip:
            obj = obj.strip()
        if callable(escape):
            obj = escape(obj)
        elif escape:
            obj = esc_tex(obj, ustrings=ustrings)

        if ustrings and isinstance(obj, bytes):
            obj = to_ustring(obj)

        return obj

    return cleanup(value)


class HTML2Tex:
    """
    Customizable html to tex parser.

    Standard usage is:
        parser = perfact.latex.HTML2Tex()
        return parser(body)
    However, it is also possible to restrict or extend the number of recognized
    tags by changing parser.trans and also to replace the handling of custom
    styles by setting parser.custom_styles to a custom function
    """

    def __init__(self):
        self.trans = {
            'p':      ('\n', '\n'),
            'br':     ('', '\\mbox{}\\newline\\mbox{}'),
            'h1':     ('\n\n\\chapter*{', '}\n'),
            'h2':     ('\n\n\\section*{', '}\n'),
            'h3':     ('\n\n\\subsection*{', '}\n'),
            'h4':     ('\n\n\\subsubsection*{', '}\n'),
            'strong': ('{\\bfseries ', '}'),
            'em':     ('{\\itshape ', '}'),

            'pre':    ('\\begin{dokbox}\\bgroup\\ttfamily ',
                       '\\egroup\\end{dokbox}'),
            'code':   ('{\\ttfamily ', '}'),

            'ol':     ('\\begin{enumerate}', '\\end{enumerate}'),
            'ul':     ('\\begin{itemize}', '\\end{itemize}'),
            'li':     ('\\item\\relax ', ''),

            'table':  self.tex_table,
            'tr':     self.tex_tr,
            'td':     self.tex_td,

            'dl':     ('\\begin{description}', '\\end{description}'),
            'dt':     ('\\item [', ']'),
            'dd':     ('', ''),

            'img':    self.tex_image,
            'span':   self.tex_span,
        }
        self.custom_styles = HTML2Tex.default_styles

    @staticmethod
    def default_styles(univ):
        '''Implement custom styles.'''
        # Replace tokens
        univ = tex_quote(univ)
        return univ

    # The following three functions provide TeX formatting for long tables
    # if we have no information whatsoever regarding the column widths.

    def tex_guess_table(self, contents, attr={}, passthrough={}):
        '''Analyze the table as well as possible to build a good profile
        for TeX.'''

        # Count the number of columns
        num_cols = 0
        # Keep track of the approximate column width
        col_width = []

        # Assume clean HTML when parsing the contents into a grid.
        rows = contents.split('<tr')[1:]
        for row in rows:
            count = 0
            cols = row.split('<td')[1:]
            for i in range(len(cols)):
                col = cols[i]
                # colspans?
                at_str, chars = col.split('>', 1)
                at = generic.attr_dict(at_str)
                span = int(at.get('colspan', '1'))
                # Find out number of characters (approx).
                chars = chars.split('</', 1)[0].strip()
                if span == 1:
                    # Append enough counters
                    col_width.extend([0] * (count+1-len(col_width)))
                    if len(chars) > col_width[count]:
                        col_width[count] = len(chars)
                else:
                    width = len(chars) / span
                    # Span is distributed
                    for s in range(span):
                        ind = count+s
                        # Append enough counters
                        col_width.extend([0] * (ind+1-len(col_width)))
                        # Perform measurement
                        if width > col_width[ind]:
                            col_width[ind] = width
                count += span
            if count > num_cols:
                num_cols = count

        # (better approach to calculating widths may be:
        #  compare the total width with an approximate page width.
        #  then decide whether you want to turn the widest columns into p{}.)

        total_width = 0
        for w in col_width:
            total_width += w

        page_widthcm = 17 - 0.7*num_cols  # approximately 17 cm of space
        page_width = (80 - 2*num_cols)  # approximately 80 chars per line.
        col_widthcm = 3  # Width of each multiline column.

        # Heuristics: total width < page width: all "l" (or other format)
        # Else: take biggest column until the remaining width - columns is
        #       below the page width

        formats = ['l', ] * num_cols
        format_colwidthcm = 0
        remaining_width = total_width
        while (remaining_width > 0 and
               ((float(remaining_width) / page_width) >
                float(page_widthcm - format_colwidthcm) / page_widthcm)):

            # Reformat broadest column
            max_ind = 0
            max_width = col_width[0]
            for i in range(1, num_cols):
                if col_width[i] > max_width:
                    max_width = col_width[i]
                    max_ind = i

            remaining_width -= max_width
            col_width[max_ind] = 0

            this_col = col_widthcm
            calculated_width = (
                float(page_width - remaining_width) /
                page_width * (page_widthcm - format_colwidthcm))

            if calculated_width > col_widthcm:
                this_col = int(calculated_width)

            formats[max_ind] = 'p{%dcm}' % this_col
            format_colwidthcm += this_col

        info = {'num_cols': num_cols,
                'col_width': col_width,
                'total_width': total_width,
                }
        passthrough['info'] = info

        # build description string.
        desc = '{|' + '|'.join(formats) + '|}'
        left, right = ('\\begin{doktabular}', '\\hline\\end{doktabular}')
        return left+desc, right

    def tex_tr(self, contents, attr={}, passthrough={}):
        left, right = '\\hline\n', '\\\\'
        passthrough['colnum'] = 0
        return left, right

    def tex_td(self, contents, attr={}, passthrough={}):
        left, right = '', ''

        # Do we have a width?
        my_width = attr.get('width')
        if my_width and passthrough.get('cm_per_px'):
            widthpx = float(my_width.replace('px', ''))
            widthcm = passthrough['cm_per_px'] * widthpx
            fmt = 'p{%.1fcm}' % widthcm
        else:
            fmt = 'l'

        if 'colspan' in attr:
            cols = int(attr['colspan'])
            left = '\\multicolumn{%d}{|%s|}{' % (cols, fmt)
            right = '}'
        else:
            cols = 1

        num_cols = passthrough.get('info', {}).get('num_cols', 1)
        colnum = passthrough.get('colnum', 0)
        colnum += cols
        passthrough['colnum'] = colnum

        if colnum < num_cols:
            right += ' & '
        return left, right

    def tex_table(self, contents, attr={}, passthrough={}):
        """
        Table layouter for HTML tables with "width" attributes.
        We expect "width" attributes in table, tr, and td!
        """
        page_widthcm = 17
        page_widthpx = 600
        page_colsepcm = 0.4

        if not attr.get('width') or attr.get('width').endswith('%'):
            # Fall back to guessing mode if no width is given or width is
            # relative
            return self.tex_guess_table(contents, attr, passthrough)

        tw = float(attr.get('width', '0').replace('px', ''))

        passthrough['cw'] = []  # Initialize column widths
        generic.html_process(
            contents, self.tex_table_parser, passthrough)

        # Calculate pixels per cm

        # We lose some space due to the column separators
        cw = passthrough['cw']
        if None in cw:
            # this happens if there are columns that are only part of colspans
            return self.tex_guess_table(contents, attr, passthrough)
        pwidthcm = page_widthcm - len(cw) * page_colsepcm

        if tw > page_widthpx:
            cm_per_px = float(pwidthcm) / tw
        else:
            cm_per_px = float(pwidthcm) / page_widthpx

        passthrough['cm_per_px'] = cm_per_px
        for i in range(len(cw)):
            cw[i] = cw[i] * cm_per_px

        # Populate passthrough for compatibility.
        info = {'num_cols': len(passthrough['cw']),
                'col_width': passthrough['cw'],
                'total_width': tw,
                }
        passthrough['info'] = info

        # Build the format
        formats = ['p{%.1fcm}' % a for a in passthrough['cw']]
        desc = '{|' + '|'.join(formats) + '|}'
        left, right = ('\\begin{longtable}[l]', '\\hline\\end{longtable}')
        return left+desc, right

    def tex_table_parser(self, data, passthrough):

        if data['tag'] == 'tr':
            passthrough['ccount'] = 0  # Initialize column counter
        elif data['tag'] == 'td':
            w = data['attrs'].get('width', '0').replace('px', '')
            if w.endswith('%'):  # we ignore relative widths
                w = None
            else:
                w = float(w)
            cs = int(data['attrs'].get('colspan', '1'))
            cc = passthrough['ccount']
            cc += cs
            cw = passthrough['cw']
            while len(cw) < cc:
                cw.append(None)
            if cs == 1 and w is not None:
                cw[cc-1] = w
            passthrough['ccount'] = cc
        return data

    def tex_span(self, contents, attr={}, passthrough={}):
        left, right = '', ''

        spancls = None
        if 'class' in attr:
            spancls = attr['class']

        # Define colors here:
        colors = {
            'textred': 'red',
            'textblue': 'blue',
        }

        texcolor = colors.get(spancls)
        if texcolor:
            left = '\\bgroup\\color{%s}' % texcolor
            right = '\\egroup{}'
            return left, right

        # Could be another type of <span>

        # Define left/right combinations here:
        specials = {
            'pagebreak': ('\\pagebreak{}', ''),
            'textsmall': ('\\bgroup\\small{}', '\\egroup'),
        }

        left, right = specials.get(spancls, ('', ''))
        return left, right

    def tex_image(self, contents, attr={}, passthrough={}):
        src = attr['src']
        opt = 'scale=0.5'
        # There's some cheating involved here: We scale PNGs down to 0.5
        # by default (because this produces good scales for
        # screenshots). We replace .png with .pdf if graphics are involved
        # which can be rendered in PDF.
        if (src.endswith('.png') and
            (src.find('/db_plot/') != -1 or
             src.find('/graphics_d/img/') != -1)):
            src = src[:-4] + '.pdf'
            opt = 'scale=1'

        # img src="data:image/png;base64,iVBO...YII="></img>
        if src.startswith('data:image/png;base64,'):
            res = ('\\begin{center}\n' +
                   '\\includegraphics['+opt+']{\n%%DATA ' +
                   to_string(src[5:])+'\n}\n' +
                   '\\end{center}\n')
            return res, ''
        if src.startswith('data:'):
            return '', ''

        res = ('\\begin{center}\n' +
               '\\includegraphics['+opt+']{\n%%PATH ' +
               to_string(src)+'\n}\n' +
               '\\end{center}\n')
        return res, ''

    # in the following section we will replace the starttag with the first and
    # the the closetag with the second list-entry:

    def tex_parser(self, data, passthrough):
        ''
        # Translate the verbatim parts
        univ = generic.html_unquote(data['before'], protect_tags=False)
        data['before'] = self.custom_styles(univ)

        # Detect (and ignore) invalid tags:
        if not data['tag'] in self.trans.keys():
            return data

        generator = self.trans[data['tag']]
        if get_type(generator) == 'tuple':
            left, right = generator
        else:
            left, right = generator(data['contents'],
                                    data['attrs'],
                                    passthrough)

        data['left'] = left
        data['right'] = right
        return data

    def __call__(self, body):
        return generic.html_process(body, self.tex_parser)


if __name__ == '__main__':
    source = open('schild.tex', 'r').read()
    print(str(barcode_pdflatex(source)))
