# coding: utf8
#
# printer.py  -  Helpers for parsing and printing labels on
#                specialized printers in graphics format.
#
# Copyright (C) 2005 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
import argparse
import re
import string
import os
import requests
import struct
import subprocess as sp
from tempfile import mkdtemp
from syslog import syslog

from .generic import cleanup_string, to_bytes


class Consumer():
    '''
    Class wrapping a subprocess that accepts input and can be used in a with
    block
    '''

    def __init__(self, cmd, **kw):
        '''
        Pass additional keyword arguments for subprocess.Popen
        '''
        self.cmd = cmd
        self.args = {'stdin': sp.PIPE, 'stderr': sp.PIPE}
        self.args.update(kw)

    def __enter__(self):
        self.proc = sp.Popen(self.cmd, **self.args)
        return self.proc.stdin

    def __exit__(self, type=None, value=None, traceback=None):
        [out, err] = self.proc.communicate()
        assert self.proc.returncode == 0, err


def cleanup_ident(name):
    '''remove unsafe characters from printer names etc.'''
    return cleanup_string(
        name,
        valid_chars=string.ascii_letters+string.digits+'-_.= ',
    )

# Set of routines for coalescing several jobs into one.


def raw_print_dir(handle, printer='label1'):
    '''Send all files in tempdir to the printer as one job.
    '''
    assert printer == cleanup_ident(printer), 'Invalid printer name'
    if not handle.startswith('/tmp/'):
        raise ValueError('Invalid dir handle')

    files = os.listdir(handle)
    files.sort()
    with Consumer(['/usr/bin/lp', '-d', printer, '-']) as fout:
        for f in files:
            fin = open(handle+'/'+f, 'rb')
            while 1:
                buf = fin.read(65536)
                if not buf:
                    break
                fout.write(buf)
            fin.close()

    return


def pdf_print(data, printer=None, cupsparam=None, copies=1, raw=False, **kw):
    '''Print PDF data directly. Keyword arguments are passed
    on to the printer as options.
    cupsparam should be a list. Earlier versions used string.
    '''
    assert printer == cleanup_ident(printer), 'Invalid printer name'
    opts = []
    if raw:
        opts += ['-o', 'raw']
    for i in kw.keys():
        opts += ['-o', cleanup_ident(i)+'='+cleanup_ident(kw[i])]
    if not cupsparam:
        cupsparam = ['-d', printer]
    if copies > 1 and 'collate' not in kw.keys():
        opts += ['-o', 'collate=true']

    for i in range(len(cupsparam)):
        cupsparam[i] = cleanup_ident(cupsparam[i])

    cmd = ['/usr/bin/lp', '-n', str(copies)] + opts + cupsparam
    with Consumer(cmd) as fout:
        if isinstance(data, (bytes, str)):
            fout.write(to_bytes(data))
        else:
            while 1:
                buf = data.read(65536)
                if not buf:
                    break
                fout.write(buf)
            data.close()


def pdf_to_png(pdf, page=0, dpmm=8, antialias=None):
    '''Render a given PDF into a b/w PNG for browser visualization.
    '''
    dpi = 25.4 * dpmm
    tmpdir = mkdtemp()

    deviceopts = '-sDEVICE=pngmono'
    if antialias:
        try:
            bits = int(antialias)
        except ValueError:
            bits = 4
        deviceopts = ('-sDEVICE=pnggray -dTextAlphaBits=%d '
                      '-dGraphicsAlphaBits=%d' % (bits, bits))

    cmd = ('gs -q -dBATCH -dNOPAUSE -r%f %s -sOutputFile=out%%04d.png -'
           % (dpi, deviceopts)
           ).split()
    with Consumer(cmd, cwd=tmpdir) as fout:
        fout.write(to_bytes(pdf))

    files = os.listdir(tmpdir)
    files.sort()
    data = open(tmpdir + '/' + files[page], 'rb').read()
    return data


def bin_to_hex(value):
    '''Convert binary string into hex string of double length'''
    out = []
    for i in bytearray(to_bytes(value)):
        out.append('%02X' % i)
    return ''.join(out)


def bin_invert(value):
    '''Invert a binary bit representation.'''
    out = bytearray([])
    for i in bytearray(to_bytes(value)):
        out.append(255-i)
    return bytes(out)


def pack_bits_encode(buf):
    '''Return a packbits encoded variant of the given buffer'''
    bitmask = 0b10000000
    current_run = 0
    current_value = False
    out = bytearray([])
    bytearr = bytearray(to_bytes(buf))

    for byte in bytearr:
        for bit in range(8):
            # in case we are to long, start a new run
            if current_run == 127:
                out.append((current_value << 7) | current_run)
                current_run = 0

            bit_value = bool(byte & (bitmask >> bit))
            if current_run == 0:
                current_value = bit_value
            elif bit_value != current_value:
                out.append((current_value << 7) | current_run)
                current_value = bit_value
                current_run = 0
            # Always increment the current run
            current_run += 1
    # Finish the last run in case it is not 0
    if current_run > 0:
        out.append((current_value << 7) | current_run)

    return bytes(out)


def read_pbm_dimensions(fin, dpmm):
    ''' Reads the dimensions from a pbm-file and leaves the
    readpointer in the dimensions line'''
    dimensions = {}

    if fin.readline().strip() != b'P4':
        raise ValueError('Wrong file format')
    while 1:
        line = fin.readline().strip()
        if line[0] == b'#':
            continue
        break
    dimensions['width'], dimensions['height'] = map(int, line.split(b' '))

    dimensions['raw_width'] = int((dimensions['width']+7)/8)
    dimensions['raw_height'] = int((dimensions['height']+7)/8)
    dimensions['raw_total'] = (dimensions['raw_width'] *
                               dimensions['raw_height'] * 8)

    dimensions['width_mm'] = float(dimensions['width']) / dpmm
    dimensions['height_mm'] = float(dimensions['height']) / dpmm

    if (dimensions['height'] % 8) != 0:
        raise ValueError('Height %d not multiple of eight' %
                         dimensions['height'])

    return dimensions


class PrintStrategy(object):

    def printout(self, printfile):
        raise Exception('Use a subclass of PrintStrategy')

    def printall(self, raw_files, tmpdir='.'):
        for f in raw_files:
            fin = tmpdir + '/' + f
            self.printout(fin)

    def optimize(self, aString):
        return aString


class PrintDirStrategy(PrintStrategy):
    '''Append one or more pages to the printer file in the temp dir
    given by handle.
    '''

    def __init__(self, handle):
        if not handle.startswith('/tmp/'):
            raise ValueError('Invalid dir handle')
        self.handle = handle

    def printall(self, raw_files, tmpdir='.'):
        fout = open(self.handle+'/out.prn', 'ab')
        for f in raw_files:
            fin = open(tmpdir + '/' + f, 'rb')
            while 1:
                buf = fin.read(65536)
                self.optimize(buf)
                if not buf:
                    break
                fout.write(buf)
            fin.close()
        fout.close()


class DirectPrintStrategy (PrintStrategy):
    ''' Class to send raw data to a printer file per file '''

    def __init__(self, printer='label1'):
        assert printer == cleanup_ident(printer), 'Invalid printer name'
        self.printer = printer

    def printout(self, printfile):
        ''' Send data to a raw printer via lp '''
        fin = open(printfile, 'rb')
        with Consumer(['/usr/bin/lp', '-d', self.printer, '-']) as fout:
            while True:
                buf = fin.read(65536)
                self.optimize(buf)
                if not buf:
                    break
                fout.write(buf)
        fin.close()


class BulkPrintStrategy (DirectPrintStrategy):
    ''' Class to send raw data to a printer in one big jobs consisting of
    defined number of different labels '''

    def printall(self, raw_files, tmpdir='.'):
        ''' collects printdata in a single file and prints when limit
        is reached '''
        # If it is only one file, the DirectPrintStrategy is faster
        if (len(raw_files) < 2):
            DirectPrintStrategy.printall(self, raw_files, tmpdir)
            return
        limit = 20
        counter = 0
        alloutfile = tmpdir+'/allout.prn'
        fout = open(alloutfile, 'ab')
        for f in raw_files:
            fin = open(tmpdir + '/' + f, 'rb')
            while 1:
                buf = fin.read(65536)
                if not buf:
                    break
                fout.write(buf)
            fin.close()
            counter = counter + 1
            if counter >= limit:
                # print and open a new alloutfile
                fout.close()
                self.printout(alloutfile)
                os.remove(alloutfile)
                fout = open(alloutfile, 'ab')
                counter = 0
        if (counter >= 0):
            fout.close()
            self.printout(alloutfile)


class BulkTestStrategy (PrintStrategy):
    ''' Class to send raw data to a testdir in one big jobs consisting of
    defined number of different labels '''

    def printout(self, printfile):
        ''' Send data to a raw printer via lp '''
        fin = open(printfile, 'rb')
        fout = open('./testneu/Ausgabe.txt', 'ab')
        while 1:
            buf = fin.read(65536)
            buf = self.optimize(buf)
            if not buf:
                break
            fout.write(buf)
        fin.close()
        fout.close()
        return

    def printall(self, raw_files, tmpdir='.'):
        ''' collects printdata in a single file and prints when limit is
        reached '''
        # If it is only one file, the DirectPrintStrategy is faster
        if (len(raw_files) < 2):
            DirectPrintStrategy.printall(self, raw_files, tmpdir)
            return
        limit = 20
        counter = 0
        alloutfile = tmpdir+'/allout.prn'
        fout = open(alloutfile, 'ab')
        for f in raw_files:
            fin = open(tmpdir + '/' + f, 'rb')
            while 1:
                buf = fin.read(65536)
                if not buf:
                    break
                fout.write(buf)
            fin.close()
            counter = counter + 1
            if counter >= limit:
                # print and open a new alloutfile
                fout.close()
                print('printing an alloutfile')
                self.printout(alloutfile)
                os.remove(alloutfile)
                fout = open(alloutfile, 'ab')
                counter = 0
        if (counter >= 0):
            fout.close()
            self.printout(alloutfile)


class TestPrintStrategy (PrintStrategy):
    ''' Does not print anything and leaves the files where they are'''

    def printall(self, raw_files, tmpdir='.'):
        print('Output-Files: ')
        for f in raw_files:
            print(tmpdir+'/'+f)


class TestPrintStrategy2 (PrintStrategy):
    ''' Does not print but moves files to ./testneu '''

    def printout(self, printfile):
        ''' Send data to a raw printer via lp '''
        # fin  = open(printfile, 'r')
        os.rename(printfile, 'testneu')
        print('file in ./testneu')
        return


class RawPDFPrinter (object):
    ''' Class for printing PDFs on a labelprinter '''

    def __init__(self, printstrategy):
        # print 'init: '+ str(self.__class__)
        self.printstrategy = printstrategy
        self.values = {}
        self.init_optimizer()

    def init_optimizer(self):
        # do nothing, there is no default optimizer
        return

    def get_header(self, **kwargs):
        '''
        Returns the header of the printer. This is a string containing
        placeholders for several values, but after these placeholders are
        inserted, it is converted to bytes.
        '''
        return ''

    def get_single_lines(self):
        return False

    def get_name(self):
        return 'noMod'

    def pbm_to_raw_fd(self, fin, fout, copies=1,
                      hoffset=0, voffset=0,
                      tearoff=None, dpmm=8,
                      rawheader=''):
        '''Convert raw PBM data into a label printer page.'''
        # Safe parameters
        copies = int(copies or 0)
        hoffset = int(hoffset or 0)
        voffset = int(voffset or 0)
        dpmm = int(dpmm or 8)

        values = self.get_newValues()
        values['hoffset'] = hoffset
        values['voffset'] = voffset
        values['copies'] = copies
        values['rawheader'] = rawheader
        dimensions = read_pbm_dimensions(fin, dpmm)
        values.update(dimensions)
        fout.write(to_bytes(self.get_header(**values) % values))

        # Write opaque binary data
        values['scanline'] = 0

        self.values = values
        self.convert_file(fin, fout, values)

        fout.write(to_bytes(self.get_footer(
            copies=copies, hoffset=hoffset,
            voffset=voffset, tearoff=None
        )))
        fin.close()
        fout.close()
        return

    def convert_file(self, fin, fout, values):
        bytes_read = 0

        while True:
            buf = fin.read(self.get_single_lines() and
                           values['raw_width'] or 65536)
            bytes_read += len(buf)

            if not buf:
                break
            if bytes_read > values['raw_total']:
                buf = buf[:-(bytes_read-values['raw_total'])]

            values['scanline'] = values['scanline'] + 1
            buf = self.convert_buf(buf, values)
            fout.write(buf)

    def printout(self, printdata, num_labels=1, batch_size=1, dpmm=8,
                 rotate='U', hoffset=16, voffset=0, tearoff=None,
                 rawheader=''):
        ''' Print out a PDF on a raw printer
        Rotate by supplying "U", "L" or "R" as "rotate" parameter.
        Flipping is supported by passing "V" or "H" as "rotate".
        '''
        # Safe parameters
        num_labels = int(num_labels or 0)
        batch_size = int(batch_size or 0)
        dpmm = int(dpmm or 8)
        hoffset = int(hoffset or 0)
        voffset = int(voffset or 0)

        dpi = 25.4 * dpmm

        tmpdir = mkdtemp()
        fileprefix = (self.get_name() + '-' + str(num_labels) +
                      '-' + str(dpmm) + '-' + str(voffset) + '-'+str(hoffset))
        fileprefix = cleanup_ident(fileprefix)
        cmd = (('gs -q -dBATCH -dNOPAUSE -r%f -sDEVICE=pbmraw'
                ' -sOutputFile=%sout%%04d.pbm -') % (dpi, fileprefix)
               ).split()
        with Consumer(cmd, cwd=tmpdir) as fout:
            fout.write(to_bytes(printdata))

        files = os.listdir(tmpdir)
        files.sort()
        raw_files = []
        count = None
        pnmflip_option_dict = {
            'U': '-r180',
            'L': '-ccw',
            'R': '-cw',
            'V': '-lr',
            'H': '-tb',
        }
        for f in files:
            if rotate:
                pnmflip_cmd = ['pnmflip']
                for rotletter in rotate:
                    pnmflip_cmd.append(pnmflip_option_dict[rotletter])

                pnmflip_cmd.append(tmpdir+'/'+f)
                proc = sp.Popen(
                    pnmflip_cmd,
                    stdout=sp.PIPE
                )
                fin = proc.stdout
            else:
                proc = None
                fin = open(tmpdir + '/' + f, 'rb')

            # batch handling
            if count is None or count >= batch_size:
                count = 0
                outfile = f[:-3]+'raw'
                raw_files.append(outfile)
            fout = open(tmpdir+'/'+outfile, 'ab')
            count += 1

            # pass this descriptor to the pbm converter
            self.pbm_to_raw_fd(fin, fout, copies=num_labels,
                               hoffset=hoffset, voffset=voffset,
                               tearoff=tearoff,
                               dpmm=dpmm, rawheader=rawheader)
            if proc:
                proc.wait()

        self.printstrategy.printall(raw_files, tmpdir=tmpdir)
        syslog('raw_pdf_print sent %d files to the printer' % len(raw_files))
        return

    def get_footer(self, copies=1, hoffset=0, voffset=0, tearoff=None):
        '''
        The get_footer function may return a string or bytes, it is transformed
        using to_bytes before being inserted into the stream.
        '''
        return b''

    def convert_buf(self, buf, values=None):
        return buf

    def get_newValues(self):
        return {}


# Subclasses for special printers

class RawPDFtoPICAPrinter(RawPDFPrinter):
    ''' PICA '''

    def pbm_to_raw_fd(self, fin, fout, copies=1, hoffset=0, voffset=0,
                      tearoff=None, dpmm=8, rawheader=''):
        ''' Convert raw PBM data into a label printer page
         PCX File variant
         Pica printers work differently. We need to supply a PCX file instead
         of raw pnm data, so first convert.'''
        # Safe parameters
        copies = int(copies or 0)
        hoffset = int(hoffset or 0)
        voffset = int(voffset or 0)
        dpmm = int(dpmm or 8)

        # Perform PCX conversion
        tmpdir = mkdtemp()
        with Consumer(['convert', '-', 'out.pcx'], cwd=tmpdir) as pcx:
            while 1:
                buf = fin.read(65536)
                if not buf:
                    break
                pcx.write(buf)

        fin.close()

        fin = open(tmpdir+'/out.pcx', 'rb')
        # Build label header and footer
        header = to_bytes(
            '\001FCCL--r0015000-\027\001AX001'
            '{voffset}06d%{hoffset}06d1\027'.format(
                voffset=voffset,
                hoffset=hoffset,
            )
        )
        footer = (b'\001FBAA00r00000000\027\n'
                  b'\001FBBA00r00001000\027\n'
                  b'\001FBC000r00000000\027\n'
                  )

        # Output:
        fout.write(header)

        # Write opaque binary data
        while 1:
            buf = fin.read(65536)
            if not buf:
                break
            fout.write(buf)
        fout.write(footer)
        fin.close()
        fout.close()
        return

    def get_name(self):
        return 'PICA'


class RawPDFtoTSPLPrinter(RawPDFPrinter):
    '''TSPL only 8dpmm'''

    def get_header(self, **kwargs):
        # TODO: Replace REFERENCE with SHIFT for modern printers
        # (SHIFT supports offsets -1in .. +1in)
        return ('SIZE %(width_mm).1f mm,%(height_mm).1f mm\n'
                'REFERENCE %(hoffset)d,%(voffset)d\n'
                'CLS\n'
                '%(rawheader)s\n'
                'BITMAP 0,0,%(raw_width)d,%(height)d,0,\n'
                )

    def get_footer(self, copies=1, hoffset=0, voffset=0,
                   tearoff=None):
        # Safe param
        copies = int(copies or 0)
        return '\nPRINT 1,%d\n' % copies

    def convert_buf(self, buf, values=None):
        return bin_invert(buf)

    def get_name(self):
        return 'TSPL'


class RawPDFtoSATOPrinter(RawPDFPrinter):
    '''SATO'''
    '''
    If labels do not show up as expected, you may need to completely
    reset the SATO printer: press FEED and LINE while powering on and
    select the default configuration.
    '''
    # Sequence '\033A\033C\033Z' repeats last label

    def get_header(self, **kwargs):
        return ('\033A\033H%(sato_hoffset)04d\033V%(sato_voffset)04d'
                '\033GB%(raw_width)03d%(raw_height)03d')

    def get_footer(self, copies=1, hoffset=0, voffset=0,
                   tearoff=None):
        # Safe param
        copies = int(copies or 0)
        return '\033Q%06d\033Z' % copies

    def get_newValues(self):
        ret = {}
        # In some cases, we need to supply offsets inside this driver
        ret['sato_hoffset'] = 350
        ret['sato_voffset'] = 0
        return ret

    def get_name(self):
        return 'SATO'


class RawPDFtoZebraPrinter(RawPDFPrinter):
    '''Zebra'''
    ''' Feeding in Zebra printers is unknown yet. '''
    # ~DY downloads graphics into UNKNOWN.GRF
    # ^PQn before ^XZ prints several labels.
    # ^XB prevents winding to the tearoff

    def get_header(self, **kwargs):
        return '~DY,B,,%(raw_total)d,%(raw_width)d,'

    def get_footer(self, copies=1, hoffset=0, voffset=0,
                   tearoff=None):
        # Safe params
        copies = int(copies or 0)
        hoffset = int(hoffset or 0)
        voffset = int(voffset or 0)

        # For cutter: footer = '\n^JSA\n^JWH\n^XA\n^MMC\n^FO%2d,%2d^XG^FS\n'
        # % (hoffset, voffset)
        footer = '\n^XA\n^MMT\n^FO%2d,%2d^XG^FS\n' % (hoffset, voffset)
        if copies > 1:
            footer += '^PQ%d\n' % copies
        # tearoff won't work this way
        if tearoff:
            footer += '~TA%03d\n' % tearoff
        footer += '^XZ\n'
        return footer

    def get_name(self):
        return 'Zebra'

    def init_optimizer(self):
        self.printstrategy.optimize = self.optimize

    def optimize(self, aString):
        ''' supress backfeed in all labels but the last'''
        print('running optimize')
        aString = re.sub(to_bytes(r'XZ[ ]*\n(?!$)'), b'XB\n^XZ\n', aString)
        return aString


class RawPDFtoZebraASCIIPrinter(RawPDFPrinter):
    '''Zebra_ASCII'''

    def get_header(self, **kwargs):
        return '~DY,A,,%(raw_total)d,%(raw_width)d,'

    def get_footer(self, copies=1, hoffset=0, voffset=0,
                   tearoff=None):
        # Safe params
        copies = int(copies or 0)
        hoffset = int(hoffset or 0)
        voffset = int(voffset or 0)

        footer = '\n^XA\n^FO%2d,%2d^XG^FS\n' % (hoffset, voffset)
        if copies > 1:
            footer += '^PQ%d\n' % copies
        if tearoff:
            footer += '~TA%03d\n' % tearoff
        footer += '^XZ\n'
        return footer

    def convert_buf(self, buf, values=None):
        return to_bytes(bin_to_hex(buf)+'\n')

    def get_name(self):
        return 'Zebra_ASCII'


class RawPDFtoEPL2Printer(RawPDFPrinter):
    '''EPL2'''

    def get_header(self, **kwargs):
        return '\nN\nGW%(hoffset)d,%(voffset)d,%(raw_width)d,%(height)d,'

    def get_footer(self, copies=1, hoffset=0, voffset=0, tearoff=None):
        return '\nP1,1\n'

    def convert_buf(self, buf, values=None):
        return bin_invert(buf)

    def get_name(self):
        return 'EPL2'


class RawPDFtoPICASLPrinter(RawPDFPrinter):
    '''PICA_SL'''
    # header is empty and can therefore be inherited from RawPDFPrinter

    def get_single_lines(self):
        return True

    def get_footer(self, copies=1, hoffset=0, voffset=0, tearoff=None):
        return ('\001FBAA00r00000000\027\n'
                '\001FBBA00r00001000\027\n'
                '\001FBC000r00000000\027\n'
                )

    def convert_buf(self, buf, values=None):
        ret = to_bytes('\001D%(scanline)04d000%(raw_width)03d' % values)
        ret = ret + buf + b'\027'
        return ret

    def get_name(self):
        return 'PICA_SL'


class RawPDFtoPCLPrinter(RawPDFPrinter):
    ''' PCL kann bisher nur die Aufloesung 12pdmm = 300dpi'''

    def get_single_lines(self):
        return True

    def get_header(self, **kwargs):
        ''' PCL preceiding the raster-image'''
        # Reset the Page and set any pagesize ST printers do not care
        ret = '\x1bE\x1b&l1A'
        # Portrait, no margins
        ret = ret + '\x1b&l0o0l0E'
        # set offsets
        ret = ret + '\x1b&l%(hoffset)du%(voffset)dZ'
        # copies
        ret = ret + '\x1b&l%(copies)dX'
        # set position for the startingpoint of the graphic
        ret = ret + '\x1b*p0x0Y'
        # set the resolution TODO: Hier muss der richtige Wert rein
        # Vorsicht, hier sind dpi gefragt und nicht dpmm
        ret = ret + '\x1b*t300R'
        # set rotation TODO: Das hier könnten wir auch benutzen
        ret = ret + '\x1b*r0F'
        # set the rasterarea
        ret = ret + '\x1b*r%(height)dt%(width)dS'
        # start the graphic at the current cursor
        ret = ret + '\x1b*r1A'
        # pjl_postamble = '''
        # \x1b%%-12345X@PJL EOJ
        # @PJL RESET
        # \x1b%%-12345X '''
        # ret = ret + pjl_postamble
        return ret

    def get_footer(self, copies=1, hoffset=0, voffset=0, tearoff=None):
        ''' PCL to end the data-transfer and reset everything '''
        # *rC is the newer version but not supported by older printers
        ret = '\x1b*rB\x1bE'
        # TODO: PJL-Footer fehlt noch
        return ret

    def convert_buf(self, buf, values=None):
        ret = to_bytes('\x1b*b%(raw_width)dW' % values)
        ret = ret + buf
        return ret

    def get_name(self):
        return 'PCL'


class RawPDFtoDatamaxASCIIPrinter(RawPDFPrinter):
    '''Datamax 7-bit ASCII Image.'''

    # presets/limitations set by the Datamax manual
    positionlimit = 9999  # max 4 decimal digits
    linelimit = 255       # max 2 hex digits
    imagename = 'LABEL'   # designator for image data field

    def __init__(self, *args, **kwargs):
        '''Add variables to keep track of duplicate lines in convert_buf().'''
        super(self.__class__, self).__init__(*args, **kwargs)
        self.linecounter = 0
        self.lastline = b''

    def get_header(self, **kwargs):
        '''Write header for Datamax ASCII image.
           Syntax:
           <STX>qA: clear memory module 'A'
           <STX>IAAFLOGO:
               I: image
               A: save to memory module 'A'
               A: data type 7bit ASCII
               F: format 7-bit D-O image load file
               LABEL: name of the following image block'''
        return '\x02qA\r\n\x02IAAF{}\r\n'.format(self.imagename)

    def get_footer(self, copies=1, hoffset=0, voffset=0, tearoff=None):
        '''Write footer for Datamax ASCII image.
           Syntax:
           FFFF: end image block
           <STX>L: begin label formatting
           1 Y 1 1 000 0000 0000 LABEL:
               1: (fixed value)
               Y: image
               1: width multiplier
               1: height multiplier
               000: (fixed value)
               0000: row position
               0000: column position
               LABEL: image name
           E: terminate formatting and print label'''
        # reset line memory and counter in case we keep using this object
        self.linecounter = 0
        self.lastline = b''
        # encoding h/v-offset as col/row position, limiting to 4 digits
        if hoffset > self.positionlimit:
            hoffset = self.positionlimit
        if voffset > self.positionlimit:
            voffset = self.positionlimit
        return 'FFFF\r\n\x02L\r\n1Y11000{0:0>4}{1:0>4}{2}\r\nE\r\n'\
               .format(voffset, hoffset, self.imagename)

    def get_single_lines(self):
        return True

    def convert_buf(self, buf, values={}):
        '''Converts a binary line to ASCII-encoded hexadecimal.
        Checks for duplicate lines and writes repeat records in this case.'''
        # limit the linecounter to 255 in order to stay within 2 hex digits
        # '<=' because we subtract 1 when writing the repeat record
        if buf == self.lastline and self.linecounter <= self.linelimit:
            self.linecounter += 1
            return b''
        else:
            repeat = b''
            if self.linecounter > 1:
                # if we encounter a new line but have seen the last line at
                # least twice, write a repeat data record containing the count;
                # subtract 1 from the linecounter, because the printer
                # interprets a repeat record not as 'print the previous line
                # x times' but as 'print the line and repeat it x times'
                repeat = to_bytes('0000FF%02x\r\n' % (self.linecounter-1))
            hexvalues = bin_to_hex(buf)
            # prefix: '80' followed by line length in bytes as 2-digit hex
            linelength = len(buf)
            if linelength > self.linelimit:
                # for line length > 255 bytes, we'd break the line prefix
                raise OverflowError('line length out of bounds', linelength)
            lineprefix = to_bytes('80%02x' % linelength)
            self.lastline = buf
            self.linecounter = 1
            return to_bytes(
                '{}{}{}{}'.format(repeat, lineprefix, hexvalues, '\r\n')
            )

    def get_name(self):
        return 'Datamax_ASCII'


class RawPDFtoMarkemLabelpoint(RawPDFPrinter):

    def get_header(self, **kwargs):
        # header for raw chunk
        chunk_header = to_bytes('=')
        chunk_header += struct.pack('>H', 12)
        # image header
        chunk_header += to_bytes('\x0c\x00')
        chunk_header += struct.pack('<H', kwargs['height'])
        chunk_header += struct.pack('<H', kwargs['width'])
        chunk_header += to_bytes('\x00\x00')
        chunk_header += struct.pack('<H', int(kwargs['raw_width']))
        chunk_header += struct.pack('<H', 0)  # row folded pack-bits

        # convert to bytes here to prevent back and forth conversion
        return to_bytes('''!V5000 <CLEAR>\n\
!C\n\
!C\n\
!y23 3\n\
!y24 %(width_mm)d\n\
!y25 %(height_mm)d\n\
!Y190 0\n\
!Y35 99\n\
!Y9 1\n\
!F G 0 %(hoffset)d %(voffset)d TR 1 1\n''' % kwargs) + chunk_header

    def get_footer(self, copies=1, hoffset=0, voffset=0, tearoff=None):
        # TODO Allow direct print, !p<number> only brings the printer
        # into a state where it will print after recieving a trigger,
        # for directly printing and applying we will need to use
        # !P<number>
        return '!F\n!p{copies}\n'.format(copies=copies)

    def get_single_lines(self):
        return False

    def convert_file(self, fin, fout, values):
        '''Read the whole file and pass it to convert_buf'''
        buf = fin.read()
        fout.write(self.convert_buf(buf, values))

    def convert_buf(self, buf, values):
        '''Before writing into the output file, perform a row-packed
        packbits algorithm. Additionally to the packbits we also define
        declare repeated rows'''
        # Perform line compression before building
        lines = [
            buf[i:i+values['raw_width']]
            for i in range(0, len(buf), values['raw_width'])
        ]

        repeated_lines = []

        for line in lines:
            if repeated_lines and line == repeated_lines[-1]['content']:
                repeated_lines[-1]['count'] = repeated_lines[-1]['count']+1
            else:
                repeated_lines.append({'count': 1, 'content': line})

        output_buffer = []

        for line in repeated_lines:
            count = line['count']
            # maximum of 255 repetitions, otherwise duplicate
            encoded_line = pack_bits_encode(line['content'])
            while count > 255:
                output_buffer.append(
                    struct.pack('>B', 255)
                    + struct.pack('>B', len(encoded_line))
                    + encoded_line
                )
                count -= 255

            if count:
                output_buffer.append(
                    struct.pack('>B', count)
                    + struct.pack('>B', len(encoded_line))
                    + encoded_line
                )

        return to_bytes('').join(
            to_bytes('=') + struct.pack('>H', len(chunk)) + chunk
            for chunk in output_buffer
        )

    def get_name(self):
        return 'Markem_Labelpoint'


classselector = {
    'TSPL':        RawPDFtoTSPLPrinter,
    'SATO':        RawPDFtoSATOPrinter,
    'Zebra':       RawPDFtoZebraPrinter,
    'Zebra_ASCII': RawPDFtoZebraASCIIPrinter,
    'EPL2':        RawPDFtoEPL2Printer,
    'PICA':        RawPDFtoPICAPrinter,
    'PICA_SL':     RawPDFtoPICASLPrinter,
    'PCL':         RawPDFtoPCLPrinter,
    'DPL_ASCII':   RawPDFtoDatamaxASCIIPrinter,
    'MARKEM_LP':   RawPDFtoMarkemLabelpoint
}


def raw_pdf_print_dir(handle, pdf, num_labels=1, mode="SATO", dpmm=8):
    '''Append one or more pages to the printer file in the temp dir
    given by handle.
    '''
    printstrategy = PrintDirStrategy(handle)
    printclass = classselector[mode](printstrategy)
    printclass.printout(pdf, num_labels=1)
    return


def raw_pdf_print(pdf, num_labels=1, printer='label1', mode="EPL2",
                  test__=None, batch_size=1, dpmm=None, rotate='U',
                  hoffset=None, voffset=None, tearoff=None,
                  rawheader='', **kw):
    '''Memory efficient raw printer utility.
    Rotate by supplying "U", "L" or "R" as "rotate" parameter.
    '''
    if dpmm is None:
        dpmm = 12
    if hoffset is None:
        hoffset = 0
    if voffset is None:
        voffset = 0

    if (test__):
        printstrategy = BulkTestStrategy()
    else:
        printstrategy = BulkPrintStrategy(printer)
    printclass = classselector[mode](printstrategy)
    printclass.printout(pdf, num_labels=num_labels,
                        dpmm=dpmm, rotate=rotate, hoffset=hoffset,
                        voffset=voffset, tearoff=tearoff,
                        rawheader=rawheader)


# ZPL preview routines

def zpl_preview_printer(zpl, printer_url):
    '''Fetch a pixel-perfect PNG label preview (bytes) from a Zebra printer.

    :param zpl: Contains the ZPL code as bytes. The caller must take care of
    choosing the correct encoding, since that can be modified with printer
    commands.
    :type zpl: bytes

    :param printer_url: Contains the HTTP URL of the printer, as reachable
    directly from this host, e.g. "http://192.168.1.2:80". The port number is
    optional, and defaults to 80.
    :type printer_url: str

    :raises requests.exceptions.ConnectionError: This happens if the printer
    is not available. That can happen when the printer goes into energy saver
    mode. [Perhaps the printer can be woken up on another port, e.g. 9100.]

    The return value is bytes, and contains the PNG image of the preview. This
    preview is a pixel-perfect preview of what the printer will be putting on
    paper.

    The printer must have the preview capability. You can check this by
    opening the printer's web interface and navigating to:
    "Directory Listing" (/dir), "Create New Script" (/zpl), "Edit" (/zpl POST)

    You may then enter ZPL example code and click on "Preview" to get a label
    preview (from URL /png).

    If this process works, then this program will work as well.

    Note that from your online tests, TEMP files may stack up on the printer.
    These are only in RAM and will be gone after the next printer reboot. You
    may also delete them by hand, using the admin password (which is 1234 by
    default on these printers).
    '''
    parameters = {
        "data": zpl,
        "dev": "R",
        "oname": "UNKNOWN",
        "otype": "ZPL",
        "prev": "Preview Label",
        "pw": "",
    }
    response = requests.post(
        url=printer_url + "/zpl",
        data=parameters,
    )
    assert response.status_code == 200, "Problem calling zpl processor"
    page = response.text

    match = re.search('<IMG SRC="(png[^"]*)"', page)
    assert match is not None, "No image found in response from printer"
    image_url = match.group(1)

    response = requests.get(url=printer_url + "/" + image_url)
    assert response.status_code == 200, "Problem calling preview image"
    return response.content


def zpl_preview_labelary(zpl, dpmm=8, widthmm=100, heightmm=150, page=0):
    '''Fetch a PNG label preview (bytes) from "labelary.com".

    :param zpl:
    :type zpl: bytes

    :param dpmm: The printer resolution in dots per millimeter. Printers
    usually have either 8 (200dpi) or 12 (300dpi). Valid values are 6, 8,
    12, and 24, default is 8.
    :type dpmm: int, optional

    :param widthmm: The label width in millimeters, default is 100.
    :type widthmm: int, optional

    :param heightmm: The label height in millimeters, default is 150.
    :type heightmm: int, optional

    :param page: In multipage prints, the page number, default is 0.
    :type page: int, optional

    The return value is bytes, and contains the PNG image of the preview. This
    preview is an approximation of what the printer will be putting on paper.

    Note that the labelary API (http://api.labelary.com) needs to be accessible
    from this host.

    The labelary API has limits which may lead to error responses:
    - Maximum 5 requests per second.
    - Maximum 50 labels per request.
    - Maximum label size of 15 x 15 inches
    - Maximum embedded object size of 5MB, for e.g. ~DU and ~DY.
    - Maximum embedded image dimensions of 2,000 x 2,000 pixels, for e.g.
      ~DG and ~DY

    Note that not all printer commands are supported by labelary, and that
    there is no guarantee that the preview is pixel-precise.
    '''
    # Enforce labelary limits
    size_limit = 15 * 25.4
    assert widthmm <= size_limit, \
        "widthmm too large, must be below {}".format(size_limit)
    assert heightmm <= size_limit, \
        "heightmm too large, must be below {}".format(size_limit)
    width = '{:.1f}'.format(widthmm / 25.4)
    height = '{:.1f}'.format(heightmm / 25.4)

    valid_dpmms = [6, 8, 12, 24]
    assert dpmm in valid_dpmms, "dpmm has to be one of {}".format(valid_dpmms)

    url = (
        'http://api.labelary.com/v1/printers/'
        '{dpmm}dpmm/labels/{width}x{height}/{page}/'
    ).format(dpmm=dpmm, width=width, height=height, page=page)

    response = requests.post(
        url,
        headers={
            'Accept': 'image/png',
            'X-Quality': 'Bitonal',  # No anti-aliasing please.
        },
        files={
            'file': zpl,
        },
    )

    assert response.status_code == 200, "Problem calling web service!"
    return response.content


def binwrapper_print():
    parser = argparse.ArgumentParser('PerFact Print Tool')
    parser.add_argument('file')
    parser.add_argument('--printer', '-p', required=True)

    subparsers = parser.add_subparsers(dest='subcommand')
    subparsers.required = True

    subparsers.add_parser('direct')

    parser_raw = subparsers.add_parser('raw')
    parser_raw.add_argument(
        '--mode', '-m', choices=classselector.keys(), required=True,
        help="The type of raw-printer"
    )
    parser_raw.add_argument('--dpmm', default=12, type=int)
    parser_raw.add_argument('--voffset', default=0, type=int)
    parser_raw.add_argument('--hoffset', default=0, type=int)
    parser_raw.add_argument('--test', action='store_true')
    args = parser.parse_args()

    with open(args.file, 'rb') as data:
        if args.subcommand == 'direct':
            pdf_print(data, args.printer)
        elif args.subcommand == 'raw':
            pdf = data.read()
            raw_pdf_print(
                pdf=pdf,
                printer=args.printer,
                mode=args.mode,
                dpmm=args.dpmm,
                voffset=args.voffset,
                hoffset=args.hoffset,
                test__=args.test
            )


if __name__ == '__main__':
    '''If run as main program, send a testing label.
    '''
    fin = open('label_test.pdf', 'rb')
    pdf = fin.read()
    fin.close()
    # raw_pdf_print(pdf, num_labels=1, printer='label01',
    # mode="PICA", dpmm=12, test__=None, voffset=14800, hoffset=10600)
    raw_pdf_print(
        pdf, num_labels=1, printer='Testlabelprinter', mode="PICA",
        dpmm=12, test__=True, voffset=0, hoffset=0)
