# -*- coding: utf-8 -*-

import six
import warnings
from .generic import get_type

try:
    from StringIO import StringIO
except ImportError:
    from io import StringIO


# Escapers for data types

def sql_boolean(value=None):
    if value:
        return 'true'
    else:
        return 'false'


def sql_char_list(indexes=None):
    '''Mimick the behaviour of the char_list() aggregate on an index list,
    returning an SQL string.
    '''
    if not indexes:
        return None
    return ', '.join(indexes)


def sql_date(value=None):
    '''Reverse the German date format into something understandable by SQL.
    TODO: Internationalization is missing here.
    '''
    return sql_datetime(value)


def sql_datetime(value=None):
    '''Reformat a German date into ISO notation, keeping the time part.
    '''
    if not value:
        return 'null'
    parts = value.split()
    if not len(parts):
        return 'null'

    value = parts[0]
    value = value.replace(',', '.')
    value = value.replace('/', '.')
    items = value.split('.')
    if len(items) != 3:
        return 'null'
    fmt = "%04d-%02d-%02d" % (int(items[2]), int(items[1]), int(items[0]))
    parts[0] = fmt
    return "'"+' '.join(parts)+"'"


def sql_datetimearray(values=None, output_type='timestamp with time zone'):
    '''Reformat a string of space separated German dates into an array of dates
    in ISO notation.

    >>> sql_datetimearray('01.10.1990 13.01.2001 24.8.2020')
    "array['1990-10-01', '2001-01-13', '2020-08-24']::timestamp with \
time zone[]"

    >>> sql_datetimearray('13.01.2001 24.8.2020', output_type='date')
    "array['2001-01-13', '2020-08-24']::date[]"

    '''
    assert isinstance(values, str) and len(values) > 0, "invalid input"
    entries = []

    for date in values.split():
        date = sql_datetime(date)
        entries.append(date)

    return "array[{}]::{}[]".format(', '.join(entries), output_type)


def sql_float(value=None):
    if value is None:
        return 'null'

    value = str(value)
    value = value.replace(',', '.')  # Usually, we use german commas
    value = value.replace(' ', '')  # Thousands separators must go
    # TODO: Accept all kinds of separators
    # TODO: Only leave the last period

    try:
        value = float(value)
    except ValueError:
        return 'null'

    return str(float(value))


def sql_int(value=None):
    if value is None:
        return 'null'

    value = str(value)
    value = value.replace(',', '.')  # Usually, we use german commas
    value = value.replace(' ', '')  # Thousands separators must go
    # TODO: Accept all kinds of separators
    # TODO: Only leave the last period

    try:
        value = int(value)
    except ValueError:
        return 'null'

    return str(int(value))


def sql_intarray(indexes=None):
    '''Return null (empty list) or a SQL representation of a bigint array.
    '''
    if get_type(indexes) == 'str':
        indexes = indexes.split()

    if not indexes:
        return 'null::bigint[]'
    indexes = [a for a in indexes if a not in ('', 0)]

    if not indexes:
        return 'null::bigint[]'

    # Convert to strings
    indexes = map(str, indexes)

    return 'array[' + ', '.join(indexes) + ']::bigint[]'


def sql_interval(value=None):
    '''Very rudimentary escaper, which works essentially like str'''
    return sql_string(value)


def sql_literal(val):
    '''Ensure there's nothing dangerous about a literal.'''
    assert val, "Empty literal!"
    return str(val).replace(';', '').replace("'", '')


def sql_string(value=None):
    '''Safety escaper for SQL strings.'''
    warnings.warn(
            'sql_string is not safe. Use psycopg.extensions.QuotedString '
            'or context.sql_string in Zope',
            DeprecationWarning)
    if not value:
        return 'null'
    value = str(value)
    return "'"+value.replace("'", "''")+"'"


def sql_text(value=None):
    '''A little more complicated than sql_string, can also handle
    backslashes.
    '''
    warnings.warn(
            'sql_text is not safe. Use psycopg.extensions.QuotedString '
            'or context.sql_text in Zope',
            DeprecationWarning)
    if not value:
        return 'null'
    value = value.replace('\r', '')
    value = value.strip()
    value = value.replace('\\', '\\\\')
    value = value.replace("'", "''")
    return "E'" + value + "'"


def sql_texthtml(value=None, html_cleanup=None):

    warnings.warn(
            'sql_texthtml is not safe. Use context.sql_texthtml in Zope',
            DeprecationWarning)
    if not value:
        return 'null'

    value = html_cleanup(value)

    value = value.replace('\\', '\\\\')
    value = value.replace("'", "''")
    return "E'" + value + "'"


def sql_time(value=None):
    return sql_string(value)


# Escapers for BLOBs

def esc_text(data):
    '''
    Escape text for use with a postgres backend.
    '''
    out = StringIO()
    for c in data:
        if c == '\\':
            out.write('\\\\')
        elif c == "'":
            out.write("\\'")
        elif c == "\r":
            continue
        elif c == "\n":
            out.write("\\n")
        elif c == '\000':
            continue
        else:
            out.write(c)
    out.seek(0)
    return out.read()


def esc_binary(data, std_string=False):
    '''
    Helper routine which escapes binary data for use
    with a PostgreSQL backend.
    '''
    if data is None:
        return None
    bs = std_string and '\\' or '\\\\'
    out = StringIO()
    for c in data:
        if six.PY2:
            oc = ord(c)
            char = c
        else:
            oc = c
            char = chr(c)
        if char == '\\':
            out.write(bs+'134')
        elif char == "'":
            out.write(bs+'047')
        elif oc < 32 or oc >= 127:
            out.write(bs+'%03o' % oc)
        else:
            out.write(char)
    out.seek(0)
    return out.read()


def unesc_binary(data):
    '''
    Backward conversion of binary data which has been
    escaped by PostgreSQL for output.
    '''
    out = StringIO()
    i = 0
    while i < len(data):
        if data[i] == '\\':
            if data[i+1] == '\\':
                out.write(data[i])
                i += 1
            elif data[i+1] in '0123456789':
                out.write(chr(int(data[i+1:i+4], 8)))
                i += 3
        else:
            out.write(data[i])
        i += 1
    out.seek(0)
    return out.read()
