import six
from argparse import Namespace

if six.PY2:
    import mock
else:
    from unittest import mock


class ModulePatches(object):
    '''Context manager for performing a series of module patches.

    Example:
    ... with ModPatches(mymodule, ['id1', 'id2', 'id3']) as mobj:
    ...     mobj.id1.return_value = 'something'
    ...     do_tests()

    This call wraps objects 'id1', 'id2', and 'id3' in mymodule with mock
    objects, exposing them in the namespace returned ("mobj" in the example).

    This is similar to nesting "with" statements using patch.object() calls.
    '''
    def __init__(self, module, object_ids):
        '''Initialize an internal dictionary, "patches" to keep track of all
        mocked objects.'''
        self.patches = {}
        self.object_ids = object_ids
        for object_id in object_ids:
            self.patches[object_id] = mock.patch.object(module, object_id)

    def __enter__(self):
        '''Perform the "__enter__()" step on all patches, returning a
        Namespace with the patched objects.'''
        self.entered = Namespace()
        for id, patch_obj in self.patches.items():
            setattr(self.entered, id, patch_obj.__enter__())
        return self.entered

    def __exit__(self, type, value, traceback):
        '''Exit all patches in reverse order, to mimick exiting nested
        "with" statements.'''
        for key in reversed(self.object_ids):
            patch_obj = self.patches[key]
            patch_obj.__exit__(type, value, traceback)
