#!/usr/bin/env python
# -*- coding: utf-8 -*-

import unittest

from .. import printer
from .mock_helper import ModulePatches, mock


class TestPrinterZPL(unittest.TestCase):

    def test_zpl_preview_printer(self):
        zpl = b'^xa^cfa,50^fo100,100^fdHello World^fs^xz'
        printer_url = 'http://192.168.1.2'

        # Patch all I/O calls
        with ModulePatches(printer, [
                    'requests'
                ]) as mobj:

            # Define return values for mocks
            resp1 = mock.Mock()
            mobj.requests.post.return_value = resp1
            resp1.status_code = 200
            resp1.text = (
                '...'
                '<IMG SRC="png?prev=Y&dev=R&oname=__TMP423&otype=PNG" '
                'ALT="R:__TMP423.PNG" WIDTH="100%" BORDER="2">'
                '...'
            )
            resp2 = mock.Mock()
            mobj.requests.get.return_value = resp2
            resp2.status_code = 200
            resp2.content = b'PNG-Image'

            # Do the call proper
            output = printer.zpl_preview_printer(zpl, printer_url)
            self.assertEqual(output, resp2.content)

            # Check how the mocks were called
            mobj.requests.post.assert_called_once_with(
                url=printer_url + '/zpl',
                data={
                    "data": zpl,
                    "dev": "R",
                    "oname": "UNKNOWN",
                    "otype": "ZPL",
                    "prev": "Preview Label",
                    "pw": "",
                },
            )
            mobj.requests.get.assert_called_once_with(
                url=printer_url + '/png?prev=Y&dev=R&oname=__TMP423&otype=PNG'
            )

    def test_zpl_preview_labelary(self):
        zpl = b'^xa^cfa,50^fo100,100^fdHello World^fs^xz'

        # Patch all I/O calls
        with ModulePatches(printer, [
                    'requests'
                ]) as mobj:

            # Define return values for mocks
            resp = mock.Mock()
            mobj.requests.post.return_value = resp
            resp.status_code = 200
            resp.content = b'PNG-Image'

            # Do the call proper
            output = printer.zpl_preview_labelary(zpl)
            self.assertEqual(output, resp.content)

            # Check how the mocks were called
            mobj.requests.post.assert_called_once_with(
                'http://api.labelary.com/v1/printers/8dpmm/labels/3.9x5.9/0/',
                headers={
                    'Accept': 'image/png',
                    'X-Quality': 'Bitonal'
                },
                files={
                    'file': b'^xa^cfa,50^fo100,100^fdHello World^fs^xz'
                },
            )
