#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# test_webservice.py  -  Test perfact.webservice
#
# Copyright (C) 2017 Jens Hinghaus <jens.hinghaus@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import requests
import unittest
import six
from .. import webservice


class webservice_tc(unittest.TestCase):

    def test_credentialsFromUrl1(self):
        ''' there are credentials '''
        url = 'http://foo:bar@perfact.de'
        creds = webservice.credentialsFromUrl(url)
        self.assertEqual('foo', creds[0], 'username not found')
        self.assertEqual('bar', creds[1], 'password not found')

    def test_credentialsFromUrl2(self):
        ''' there are no credentials '''
        url = 'http://www.perfact.de'
        creds = webservice.credentialsFromUrl(url)
        self.assertIsNone(creds, 'creds should be None')

    def test_removeCredsWith(self):
        ''' remove credentials '''
        url = 'https://foo:bar@perfact.de'
        reduced_url = webservice.removeCredentialsFromUrl(url)
        self.assertEqual(reduced_url, 'https://perfact.de')

    def test_removeCredsWithout(self):
        ''' remove credentials '''
        url = 'https://perfact.de'
        reduced_url = webservice.removeCredentialsFromUrl(url)
        self.assertEqual(reduced_url, url)

    def test_translate_xml(self):
        xml_data = '<doc><foo>foo</foo><bar>baz</bar></doc>'
        res = webservice.translate_xml(xml_data)
        self.assertIn('doc', res, 'Missing root element')
        self.assertEqual(res['doc'][0], {'foo': 'foo', 'bar': 'baz'})

    def test_translate_xml_delete_ns(self):
        xml_data = '<ns:doc><ns:foo>foo</ns:foo><ns:bar>baz</ns:bar></ns:doc>'
        res = webservice.translate_xml(xml_data, delete_ns=True)
        self.assertIn('doc', res, 'Missing root element')
        self.assertEqual(res['doc'][0], {'foo': 'foo', 'bar': 'baz'})

    def test_timeout(self):
        # TODO: This needs some mocking, it will fail in an environment without
        # name resolution
        self.assertRaises(
            requests.exceptions.Timeout,
            webservice.getWS,
            'https://ema.perfact.de',
            requests_args={'timeout': 0.001},
        )

    def test_getWS_args(self):
        """
        Make sure arguments like authtype, which are used by the decorator
        sessionEnabled but throw an error if passed to requests, can be used.
        """
        # TODO: Mocking!
        self.assertRaises(
            requests.exceptions.Timeout,
            webservice.getWS,
            'https://ema.perfact.de',
            authtype='basic',
            requests_args={'timeout': 0.001},
        )

    # Disabled due to external dependency that currently does not answer
    def disabled_test_soap_retries(self):
        """
        Check that SOAP call that hits a "service unavailable" causes a retry
        error, as defined by the retry logic in call_soapservice
        """
        self.assertRaises(
            requests.exceptions.RetryError,
            webservice.call_soapservice,
            msg='This is not SOAP',
            endpoint='https://SOAPUI:Password@httpbin.org/status/503',
            rawoutput=True,
            max_retries=3,
            timeout=5
        )

    # Disabled due to external dependency that currently does not answer
    def disabled_test_soap_fails(self):
        """
        Check that SOAP call that hits 404 raises an actual server exception,
        rather than attempting retries
        """
        six.assertRaisesRegex(
            self,
            Exception,
            'Server returned status 404',
            webservice.call_soapservice,
            msg='This is not SOAP',
            endpoint='https://SOAPUI:Password@httpbin.org/status/404',
            rawoutput=True,
            max_retries=3,
            timeout=5
        )
