#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# webservice.py  -  Tools for webservices
#
# $Revision: 1.0 $
#
# Copyright (C) 2016 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

import requests
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry
import re
import xml.parsers.expat
import netrc
import six

try:
    from requests_ntlm import HttpNtlmAuth
except ImportError:
    def ntmlImportMissing(*args, **kw):
        raise Exception('You need ntlm-auth and requests-ntlm to use this')
    HttpNtlmAuth = ntmlImportMissing

try:
    from msal import ConfidentialClientApplication
except ImportError:
    def msalMissing(*args, **kw):
        raise Exception('You need python3-msal to use this')
    tokenFromMSAL = msalMissing
else:
    def tokenFromMSAL(client_id=None, authority=None, client_credential=None,
                      proxies=None, scopes=None):
        ''' returns an authentication token from a MSAL server'''
        app = ConfidentialClientApplication(
            client_id=client_id,
            authority=authority,
            client_credential=client_credential,
            proxies=proxies
        )
        token_raw = app.acquire_token_for_client(scopes=scopes)
        return token_raw


from .generic import to_string, to_bytes, to_ustring


def credentialsFromNetRC(path, machine):
    ''' Get username and password from a netrc file and return it
        as dictionary.
        Example for path: "/home/zope/.netrc-grafana-credentials"
        Use the value after machine in the netrc file for the
        machine parameter value. '''
    netrc_content = netrc.netrc(path)
    username, account, password = netrc_content.authenticators(machine)
    return {'username': username, 'password': password}


def credentialsFromUrl(url):
    ''' extracts a tuple of (username,password) from a url '''
    m = re.match(r"^https?://(.+):(.+)@", url)
    if m:
        return (m.group(1), m.group(2))
    else:
        return None


def removeCredentialsFromUrl(url):
    ''' return the given url without credentials '''
    return re.sub(r'://.+:.+@', '://', url)


def normalizeDict(anyDict):
    ''' get a CaseInsensitiveDict and return a normal dict '''
    result = {pair[0]: pair[1] for pair in anyDict.items()}
    return result


# Session Handling
def createSession(login_url, login_data):
    ses = requests.Session()
    res = ses.post(login_url, login_data)
    if res.status_code != requests.codes.ok:
        raise Exception(
            'Login failed with Status Code {}'.format(res.status_code))
    return ses


def closeSession(session, logout_url):
    session.post(logout_url)


def sessionEnabled(func):
    ''' Decorator for session handling'''
    def deco(*args, **kwargs):
        if ('login_url' in kwargs) and ('login_data' in kwargs):
            ses = createSession(login_url=kwargs['login_url'],
                                login_data=kwargs['login_data'])
            kwargs['session'] = ses
            res = func(*args, **kwargs)
            if ('logout_url' in kwargs):
                closeSession(session=ses,
                             logout_url=kwargs['logout_url'])
        else:
            res = func(*args, **kwargs)
        return res
    return deco


# Authentication Method
authclassMapping = {
    'basic': requests.auth.HTTPBasicAuth,
    'digest': requests.auth.HTTPDigestAuth,
    'ntlm': HttpNtlmAuth,
    'no_auth': 'no_auth',
}


def authtypeChoosable(func):
    ''' Decorator for choosing the authtype.
    basic, digest and token are possible values '''
    def deco(*args, **kwargs):
        if (('authtype' in kwargs) and (kwargs['authtype'] == 'token') and
                ('token_url' in kwargs)):
            # get token and prepare the header
            token = getText(kwargs['token_url'])
            token = token.strip('"')
            kwargs['headers']['Authorization'] = "Bearer %s" % (token)
        elif ('authtype' in kwargs) and ('auth' in kwargs):
            authclass = authclassMapping.get(kwargs['authtype'], None)
            if authclass is None:
                raise Exception('Unknown Authtype %s' % kwargs['authtype'])
            if authclass == 'no_auth':
                # not recommended, but supported
                return func(*args, **kwargs)
            if len(kwargs['auth']) < 2:
                msg = 'Incomplete Credentials. Username or Password missing'
                raise ValueError(msg)
            kwargs['auth'] = authclass(kwargs['auth'][0], kwargs['auth'][1])
        return func(*args, **kwargs)
    return deco


@authtypeChoosable
@sessionEnabled
def getWS(url, headers=None, params=None, auth=None,
          session=None, requests_args=None, **kwargs):
    '''
    sends a GET request
    params is a dict that gets encoded in the url
    auth is a tuple like ('user','password')
    requests_args may be a dict with additional arguments for session.get
    '''
    if session is None:
        session = requests
    res = session.get(url=url, headers=headers,
                      params=params, auth=auth, verify=False,
                      **(requests_args or {}))
    return res


def getJSON(url, headers=None, params=None, auth=None,
            session=None, **kwargs):
    '''
    sends a GET request
    params is a dict that gets encoded in the url
    auth is a tuple like ('user','password')
    any additional arguments are passed through to getWS
    assumes json output from the ws
    returns a dict
    '''
    res = getWS(url=url, headers=headers, params=params,
                auth=auth, session=session, **kwargs)

    data = res.json()
    return data


def getXML(url, headers=None, params=None, auth=None,
           session=None, **kwargs):
    '''
    sends a GET request
    params is a dict that gets encoded in the url
    auth is a tuple like ('user','password')
    any additional arguments are passed through to getWS
    assumes XML output from the ws
    returns a dict
    '''
    res = getWS(url=url, headers=headers, params=params,
                auth=auth, session=session, **kwargs)
    data = translate_xml(
        data=res.text, delete_ns=True, add_attributes=True)
    return data


def getText(url, headers=None, params=None, auth=None,
            session=None, **kwargs):
    '''
    sends a GET request
    params is a dict that gets encoded in the url
    auth is a tuple like ('user','password')
    any additional arguments are passed through to getWS
    returns a text from the ws
    '''
    res = getWS(url=url, headers=headers, params=params,
                auth=auth, session=session, **kwargs)
    return res.text


@authtypeChoosable
@sessionEnabled
def postWS(url, headers=None, data=None, json=None, auth=None,
           session=None, raisemode=True, requests_args=None,
           response_limit=256, **kwargs):
    """Send a POST request.

    data is a dict that gets form-encoded or a string that is send as it is
    json is a dict that gets json-encoded as an alternative to data
    auth is a tuple like ('user','password')
    requests_args may be a dict with additional arguments for session.post
    """
    if session is None:
        session = requests
    if data is not None:
        data = to_bytes(data)
    res = session.post(url=url, headers=headers, data=data, json=json,
                       auth=auth, verify=False, **(requests_args or {}))
    if not res and raisemode:
        msg = to_ustring(res.text).encode('ascii', 'ignore')
        if response_limit is not None:
            msg = msg[:response_limit]
        raise Exception('Server returned status {0}: {1}'.format(
            res.status_code,
            to_string(msg)
        ))
    else:
        return {
            'status': res.status_code,
            'data': res.text,
            'object': res,
            'headers': normalizeDict(res.headers)
        }


@authtypeChoosable
@sessionEnabled
def patchWS(url, headers=None, data=None, json=None, auth=None,
            session=None, requests_args=None, **kwargs):
    """Send a PATCH request.

    data is a dict that gets form-encoded or a string that is send as it is
    json is a dict that gets json-encoded as an alternative to data
    auth is a tuple like ('user','password')
    requests_args may be a dict with additional arguments for session.patch
    """
    if session is None:
        session = requests
    if data is not None:
        data = to_bytes(data)
    res = session.patch(url=url, headers=headers, data=data,
                        json=json, auth=auth, verify=False,
                        **(requests_args or {}))
    if not res:
        raise Exception('Server returned status {0}: {1}'.format(
            res.status_code, res.text))
    else:
        return {
            'status': res.status_code,
            'data': res.text,
            'object': res
        }


@authtypeChoosable
@sessionEnabled
def putWS(url, headers=None, data=None, json=None, auth=None,
          session=None, requests_args=None, **kwargs):
    """Send a PUT request.

    data is a dict that gets form-encoded or a string that is send as it is
    json is a dict that gets json-encoded as an alternative to data
    auth is a tuple like ('user','password')
    requests_args may be a dict with additional arguments for session.put
    """
    if session is None:
        session = requests
    if data is not None:
        data = to_bytes(data)
    res = session.put(url=url, headers=headers, data=data,
                      json=json, auth=auth, verify=False,
                      **(requests_args or {}))
    if not res:
        raise Exception('Server returned status {0}: {1}'.format(
            res.status_code, res.text))
    else:
        return {
            'status': res.status_code,
            'data': res.text,
            'object': res
        }


@authtypeChoosable
@sessionEnabled
def deleteWS(url, headers=None, auth=None, session=None, requests_args=None,
             **kwargs):
    """Send a DELETE request.

    auth is a tuple like ('user','password')
    requests_args may be a dict with additional arguments for session.delete
    """
    if session is None:
        session = requests
    res = session.delete(url=url, headers=headers,
                         auth=auth, verify=False,
                         **(requests_args or {}))
    if not res:
        raise Exception('Server returned status {0}: {1}'.format(
            res.status_code, res.text))
    else:
        return {
            'status': res.status_code,
            'data': res.text,
            'object': res
        }


# SOAP

class GenericXML:
    '''Build a parser for simple hierarchical dictionaries.
    no attributes are allowed.
    an element contains either data or other elements.
    residual data is in the 'data' dictionary element.
    '''

    def __init__(self, encoding='utf-8', delete_ns=True,
                 strip_spaces=True, ignore_attrs=True):
        # Set this to the desired encoding.
        self.enc = encoding
        self.del_ns = delete_ns
        self.strip_spaces = strip_spaces
        self.ignore_attrs = ignore_attrs

        self.p = xml.parsers.expat.ParserCreate()
        self.p.StartElementHandler = self.start_element
        self.p.EndElementHandler = self.end_element
        self.p.CharacterDataHandler = self.char_data
        self.p.buffer_text = True
        return

    def parse(self, data):
        '''Send data through parser and return finished dictionary.
        '''
        # Element stack is initialized with the root dictionary.
        self.stack = [('', {})]
        data = to_string(data)
        self.p.Parse(data, 1)
        return self.stack[0][1]

    def start_element(self, name, attrs):
        '''Push a new element on the stack and start
        its sub-dictionary.
        '''
        if self.del_ns:
            name = name.rsplit(':', 1)[-1]

        if self.enc:
            name = name.encode(self.enc)

        d = {}                       # new dictionary

        if attrs and not self.ignore_attrs:
            d['__attrs__'] = attrs

        if name not in self.stack[-1][1]:
            self.stack[-1][1][name] = []
        if not isinstance(self.stack[-1][1][name], list):
            # Something nasty happened: a tag containing only data has
            # been repeated!
            # We ignore such mistakes by pushing the tag onto the
            # validation stack
            self.stack.append((name, {'__ignore__': ''}))
            return
        self.stack[-1][1][name].append(d)  # add new dictionary to parent
        self.stack.append((name, d))  # push reference on stack.
        return

    def end_element(self, name):
        '''Pop the named element, checking for character contents.
        '''
        if self.del_ns:
            name = name.rsplit(':', 1)[-1]

        if self.enc:
            name = name.encode(self.enc)

        e = self.stack.pop()
        assert e[0] == name, "Closing tag does not match opening tag!"
        d = e[1]  # validate popped dictionary (reference!)
        if '__ignore__' in d:
            # Emergency measure for broken XML
            return
        if not d:
            del self.stack[-1][1][name]
        if len(d.keys()) == 1 and '__data__' in d.keys():
            # only '__data__' present: replace with contents
            self.stack[-1][1][name] = d['__data__']
        return

    def char_data(self, data):
        '''Add a '__data__' entry to the dictionary.
        '''
        if self.enc:
            val = data.encode(self.enc)
        else:
            val = data

        if self.strip_spaces:
            val = val.strip()
        if not val:
            return
        d = self.stack[-1][1]
        if '__data__' in d:
            d['__data__'] += val
        else:
            d['__data__'] = val
        return


def translate_xml(data, delete_ns=False,
                  strip_spaces=True,
                  add_attributes=False):
    '''Simple dictionary translator.'''
    encoding = None if not six.PY2 else 'utf-8'
    p = GenericXML(delete_ns=delete_ns, strip_spaces=strip_spaces,
                   ignore_attrs=not add_attributes, encoding=encoding)
    d = p.parse(data)
    del p
    return d


def soapConvertTo12(msg):
    ''' replace a 1.1 envelope namespace with a 1.2 envelope namespace '''
    return msg.replace('http://schemas.xmlsoap.org/soap/envelope/',
                       'http://www.w3.org/2003/05/soap-envelope')


def soapConvertTo11(msg):
    ''' replace a 1.2 envelope namespace with a 1.1 envelope namespace '''
    return msg.replace('http://www.w3.org/2003/05/soap-envelope',
                       'http://schemas.xmlsoap.org/soap/envelope/')


soap_version_converter = {
    '1.1': soapConvertTo11,
    '1.2': soapConvertTo12
}


def call_soapservice(msg, endpoint, soapaction=None, add_headers=None,
                     credentials=None, version='1.2', authtype='basic',
                     response_headers=False, raisemode=True, rawoutput=False,
                     max_retries=0, timeout=300):
    ''' cleaner replacement for the old generic call'''
    if add_headers is None:
        add_headers = {}
    if credentials is None:
        credentials = credentialsFromUrl(endpoint)
        # removeCredentialsFromUrl
        endpoint = removeCredentialsFromUrl(endpoint)
    if type(credentials) is dict:
        credentials = (credentials['username'], credentials['password'])

    assert version in ['1.1', '1.2'], "Unrecognized version string"

    msg = soap_version_converter[version](msg)

    header_sets = {
        '1.1': {
            'Content-Type': 'text/xml; charset=utf-8',
            'SOAPAction': '""'
        },
        '1.2': {
            'Content-Type': 'application/soap+xml; charset=utf-8'
        },
    }

    # Augment and quote the headers
    my_headers = header_sets[version]
    my_headers.update(add_headers)

    # Create configured session
    session = requests.Session()
    retry_strategy = Retry(
        total=max_retries,
        # Retry on given error codes: request timeout, too many requests,
        # bad gateway, service unavailable, gateway timeout
        status_forcelist=[408, 429, 500, 502, 503, 504],
        # Retry only for POST, since that's what we're sending
        method_whitelist=['POST'],
        # Without this, any actual error is masked by the MaxRetryError
        raise_on_status=False,
    )
    adapter = HTTPAdapter(max_retries=retry_strategy)
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    # Perform the query
    response = postWS(url=endpoint, headers=my_headers, data=msg, json=None,
                      auth=credentials, session=session, authtype=authtype,
                      raisemode=raisemode, requests_args={'timeout': timeout},
                      response_limit=None)
    response_object = response['object']
    headers = response['headers']

    response_object.encoding = 'utf-8'
    data = response_object.text.encode('utf-8', 'replace')

    if rawoutput:
        d = data
    else:
        # Unpack XML
        d = translate_xml(data, delete_ns=True, strip_spaces=True)
    if response_headers:
        return d, headers
    return d


def call_odataservice(payload, endpoint, add_headers=None,
                      credentials=None, authtype=None):
    ''' cleaner replacement for the old generic call'''

    if add_headers is None:
        add_headers = {}

    # try so get credentials from url
    if authtype is None and credentials is None:
        credentials = credentialsFromUrl(endpoint)
        endpoint = removeCredentialsFromUrl(endpoint)

    # try to get credential from netrc
    if authtype is None and credentials is None:
        # get host from endpoint string
        host = endpoint
        if host.startswith('https'):
            host = host.replace('https://', '').split('/')[0]
        elif host.startswith('http'):
            host = host.replace('http://', '').split('/')[0]

        # get auth from file .netrc-odata, make sure this file exists
        # and contains the login informations for the host.
        # This feature exists only for compatibility reasons.
        # Do not place credentials in netrc unless there is no other way
        knownhosts = netrc.netrc("/home/perfact/.netrc-odata")
        user, account, password = knownhosts.authenticators(host)
        credentials = {}
        credentials['username'] = user
        credentials['password'] = password

    headers = {'Content-Type': 'application/json'}
    # Augment and quote the headers
    headers.update(add_headers)

    # Perform the query
    data = postWS(url=endpoint, headers=headers, data=payload,
                  auth=(credentials['username'], credentials['password']),
                  session=requests)['object']

    data.encoding = 'utf-8'
    data = data.text.encode('utf-8', 'replace')

    if not data:
        raise AssertionError(
            "Got no data from Server.")

    return data
