# -*- coding: utf-8 -*-
#
# zopeinterface.py  -  Interface with PerFact's standard Zope installation
#
# Copyright (C) 2016 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
'''
This file allows integrating calls to Zope into Python code:

DEPRECATION WARNING: This interface will not be supported in Python3!

NOTE: Long integers cannot be transmitted!

NOTE: Strings are automatically converted server-side into Unicode if
they use the 8th bit1

NOTE: Dictionary keys must be strings!


>>> zopeinterface.logic('EMA/echo', value='Hallo Welt')  # doctest: +SKIP
'Hallo Welt'

>>> zopeinterface.logic('EMA/echo', value='Hallö Welt')  # doctest: +SKIP
u'Hall\\xf6 Welt'

>>> zopeinterface.logic('EMA/echo', value=1234)  # doctest: +SKIP
1234

>>> zopeinterface.logic(
...     'EMA/echo', value=[1234, 5678, 'Hallo Welt']
... )  # doctest: +SKIP
[1234, 5678, 'Hallo Welt']

>>> zopeinterface.logic(
...     'EMA/echo', value={'a': 'b', '12': 34}
... )  # doctest: +SKIP
{'a': 'b', '12': 34}

'''


# For configuring the URL with the environment
import os
# For using XMLRPC
import xmlrpclib
# For using requests directly
import requests


class ZopeInterface:
    '''A class which sets up a server connection in order to call methods
    within "logic_d".
    '''

    def __init__(self, base_url):
        self.base_url = base_url
        self.server = xmlrpclib.ServerProxy(self.base_url)

    def logic(self, method, **kwargs):
        '''Call a server-side logic function by passing the keyword arguments
        to the appropriate "execute" wrapper.'''
        kwargs['methodName'] = method
        return self.server.logic(kwargs)


# Default URL (site dependent)
default_url = (
    'http://test:1234@localhost:9081/PerFact/DB_Utils/zI18N/zLayout/'
    'zDB/Template'
)
zope_url = os.environ.get('ZOPELOGIC_URL', default_url)

# Singleton instance
zopeinterface = ZopeInterface(zope_url)

# API using requests


def send_post(url, data=None, user_cookie=None, auth=None):
    '''Call the "url" with a POST request containing "data". Send
    "user_cookie" as an authorization cookie.

    >>> send_post('http://localhost/Template/logic/EMA/echo',
    ...           {'value': u'Hello world'}, auth=('test', '1234')
    ... )  # doctest: +SKIP
    {'status_code': 200, 'text': u'Hello world'}
    '''
    is_localhost = True
    jar = requests.cookies.RequestsCookieJar()
    if user_cookie:
        jar.set('__user_cookie', user_cookie, path='/')
    res = requests.post(
        url, data,
        cookies=jar,
        auth=auth,
        verify=(not is_localhost),
        timeout=10)
    return {
        'status_code': res.status_code,
        'text': res.text,
    }
