#!/usr/bin/python3

import pandas as pd
import requests
import psycopg2
from psycopg2 import sql
import argparse

url = 'https://www.gs1.org/standards/barcodes/application-identifiers'

insert_template = '''
insert into ean128 (
    ean128_author,
    ean128_modtime,
    ean128_ai,
    ean128_length,
    ean128_name,
    ean128_caption,
    ean128_format,
    ean128_isdigit,
    ean128_decplaces
) values {values}
on conflict do nothing;
'''

values_template = '''(
    {author},
    now(),
    {ai},
    {length},
    {name},
    {caption},
    {format},
    {isdigit},
    {decplaces}
)'''

if __name__ == '__main__':
    parser = argparse.ArgumentParser(
        description='Tool for generating sql-code to fill the ean128-table'
    )

    parser.add_argument(
        '--conn-string', '-c', type=str,
        default='dbname=perfactema user=zope'
    )

    parser.add_argument(
        '--out', '-o', type=str,
        default=None
    )

    args = parser.parse_args()
    conn_string = args.conn_string

    conn = psycopg2.connect(conn_string)

    template = sql.SQL(insert_template)

    response = requests.get(url).text

    values = []
    # Just convert every column in result to string
    res = pd.read_html(
        url, displayed_only=False, converters={i: str for i in range(6)}
    )[0].to_dict('records')

    for item in res:
        ai = item['AI']
        # Just ignore AIs with 9, as they are company specific and
        # have to be defined manually anyways!
        if ai.startswith('9'):
            continue
        name = item['Description'][0:80]
        # For whatever reason this might become an NaN float
        if type(item['Data title']) != float:
            caption = item['Data title'][0:40]
        else:
            caption = ''
        format = item['Format']
        isdigit = True
        # There are AIs which have a fixed length but still
        # require FNC1
        variable_length = item['FNC1 required?'] == 'Yes'
        length = 0
        decplaces = 0

        code_parts = format.split('+')

        for part in code_parts[1:]:
            if 'X' in part:
                isdigit = False
            if '..' in part:
                variable_length = True
                continue
            length += int(part[1:])

        if variable_length:
            length = None

        if len(ai) == 4 and ai[0] == '3':
            decplaces = int(ai[3])

        format = format.replace('X', 'an').replace('N', 'n')

        values.append(
            sql.SQL(
                values_template
            ).format(
                author=sql.Literal('perfact'),
                ai=sql.Literal(ai),
                length=sql.Literal(length),
                name=sql.Literal(name),
                caption=sql.Literal(caption),
                format=sql.Literal(format),
                isdigit=sql.Literal(isdigit),
                decplaces=sql.Literal(decplaces)
            )
        )

    sql_string = template.format(
        values=sql.SQL(',').join(values)
    ).as_string(conn)

    if args.out:
        with open(args.out, 'w') as f:
            f.write(sql_string)
    else:
        print(sql_string)
