#!/usr/bin/python
'''
This script will communicate with the balance.
The given command will be transfered and in return the result will be returned.

Several commands can be processed.
'''

import socket
import time
import sys
import syslog
from perfact.generic import to_bytes, to_string


def readdata(testing=False):
    start = time.time()
    data = b''
    if not testing:
        while True:
            c = soc.recv(1)
            if c in b'\r\n':
                break
            data += c
    else:
        # a result looks like:
        # ss,wtwt,wwwwwwww,uu
        data = b'ST,GS,123.456,KG'

    data = to_string(data)

    result = []

    # parsing
    lines = data.split('\n')
    for line in lines:
        fulline = line
        g = False
        o = False
        line.rstrip('\r')
        line.lstrip(' ')
        if '+' in line:
            line = line.split('+')[1]

        if 'ST' in line:
            o = True
            line = line.replace('ST,', '')
        else:
            line = line.replace(line[0:3], '')

        if 'GS' in line:
            g = True
            line = line.replace('GS,', '')
        else:
            line = line.replace(line[0:3], '')

        try:
            weight = float(line.split(',')[-2])
        except ValueError:
            weight = None
        result.append({
            'weight': line.strip(' '),
            'balanced': g,
            'signal_o': o,
            'fulline': fulline,
            'fweight': weight,
            'unit': line.split(',')[-1]
        })

    # always get several values
    # make sure that they are all equal
    for entry in result:
        balanced = True
        if entry != result[0]:
            balanced = False
    if not balanced:
        result[-1]['balanced'] = None

    result = result[-1]
    end = time.time() - start
    syslog.syslog('convert the data took: ' + str(end))
    return result


def communicate_precisa(command='READ', value=''):
    # open up the connection to the balance
    start = time.time()
    if command not in ['READ', 'READPERMANENT', 'TARA',
                       'RESET', 'RESTART', 'NULL', 'ZERO']:
        return {'error': 'no valid command'}

    if command == 'RESTART':
        soc.send(to_bytes('OFF\r\n'))
        soc.send(to_bytes('ON\r\n'))

    if command == 'RESET':
        soc.send(to_bytes('C\r\n'))

    sleeptime = 0.1
    if command == 'READPERMANENT':
        while True:
            soc.send(to_bytes('PRT\r\n'))
            data = readdata()
            print(data)
            time.sleep(sleeptime)

    if command == 'READ':
        start = time.time()
        soc.send(to_bytes('READ\r\n'))
        data = readdata()
        syslog.syslog('read from balance: ' + str(time.time() - start))
        return data

    if command == 'NULL':
        start = time.time()
        soc.send(to_bytes('Z\r\n'))
        data = readdata()
        syslog.syslog('read from balance: ' + str(time.time() - start))
        return data

    if command == 'ZERO':
        start = time.time()
        soc.send(to_bytes('TARE\r\n'))
        data = readdata()
        syslog.syslog('read from balance after ZERO: ' +
                      str(time.time() - start))
        return data

    if command == 'TARA':
        if value:
            soc.send(to_bytes('T'+str(value)+'\r\n'))
        else:
            soc.send(to_bytes('T\r\n'))
        soc.send(to_bytes('PRT\r\n'))
        data = readdata()
        return data


if __name__ == '__main__':
    '''If called directly (e.g. via ssh for remote readouts) pass
    command line options as command and value.
    '''
    start = time.time()
    reset = 0
    debug = 0
    hostname = ''
    port = 0

    if len(sys.argv) < 4:
        print(
            "Missing argument. Usage: precisa.py host port command [ value ]")
        sys.exit()

    me = sys.argv.pop(0)
    hostname = sys.argv.pop(0)
    port = int(sys.argv.pop(0))
    command = sys.argv.pop(0)
    try:
        value = sys.argv.pop(0)
    except IndexError:
        value = ''

    for token in sys.argv:
        if token == 'debug':
            debug += 1

    socket.setdefaulttimeout(1)
    soc = socket.socket()
    soc.connect((hostname, port))
    soc.send(to_bytes('P0\r\n'))

    b = communicate_precisa(command=command, value=value)
    syslog.syslog('main called: took: ' + str(time.time() - start))
    print(b)
