#
# barcode.py  -  Barcode related utilities
#
# Copyright (C) 2014 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

# For datamatrix renderer
import subprocess

# For QR codes
import qrcode as qrcode_mod

# For making short doctests
import hashlib

from . import generic


def quickhash(data):
    bdata = generic.to_bytes(data)
    return hashlib.md5(bdata).hexdigest()


def datamatrix(data, encoding='a', **kw):
    '''
    Create data matrix barcode.
    >>> sizes, image = datamatrix(data=b'test')  # doctest: +SKIP
    >>> quickhash(image)  # doctest: +SKIP
    '3fe997cf2b3d8ae5cbed6caa9242b71a'
    >>> sizes  # doctest: +SKIP
    [12, 12]
    '''
    assert encoding in 'bfactxe8', "Unknown encoding"
    cmd = ('dmtxwrite --margin=1 --module=1 --encoding=%s '
           '--format=PGM' % encoding)
    pipe = subprocess.Popen(cmd, shell=True,
                            stdin=subprocess.PIPE,
                            stdout=subprocess.PIPE,
                            close_fds=True)
    pipe.stdin.write(generic.to_bytes(data))
    pipe.stdin.close()
    # File could be P5 or P6, depending on dmtxwrite version
    filetype = pipe.stdout.readline().strip()
    assert filetype in (b'P5', b'P6'), "Unknown PGM file type"
    sizes = pipe.stdout.readline()
    sizes = [int(a) for a in sizes.split()]
    # Max gray value can be 65535 or 255, depending on dmtxwrite version
    maxgray = pipe.stdout.readline().strip()
    assert maxgray.isdigit(), "Max gray value not integer"
    maxgray = int(maxgray)
    assert maxgray in (255, 65535), "Invalid max gray value"
    pixelsize = (1 if maxgray == 255 else 2)
    out = []
    linelength = (sizes[0]) * pixelsize
    pipe.stdout.read(linelength)  # Drop first line
    for i in range(sizes[1] - pixelsize):
        line = pipe.stdout.read(linelength)
        linestr = ''
        for j in range(sizes[0] - pixelsize):
            pixel = line[(j+1) * pixelsize]
            if isinstance(pixel, str):
                # Bytes are handles as char in Python2
                pixel = ord(pixel)
            if pixel == 0:
                linestr += '#'
            else:
                linestr += ' '
        out.append(linestr)
    pipe.stdout.read()
    image = '\n'.join(out)
    return sizes, image


def qrcode(data, level=0, mode='txt', **kw):
    '''
    Create QR code data.
    >>> size, image = qrcode(data='Testing', mode='txt')
    >>> len(image) == 21 * 22 - 1
    True
    >>> size
    [21, 21]
    '''
    c = qrcode_mod.QRCode(error_correction=level,
                          border=0)
    c.add_data(data)
    c.best_fit()
    c.make()

    if mode == 'txt':
        data = c.get_matrix()
        sizey = len(data)
        out = []
        for line in data:
            sizex = len(line)
            chars = [('#' if a else ' ') for a in line]
            out.append(''.join(chars))
        return [sizex, sizey], '\n'.join(out)

    assert False, "mode not implemented: " + mode


def code_39(code, **kw):
    '''
    Encode using Code 39.
    >>> code_39('TST123')
    '1211212111111121221111211122111111212211211211112111221111212122111111\
121121211'
    '''

    codeset = {
        '0': '111221211',
        '1': '211211112',
        '2': '112211112',
        '3': '212211111',
        '4': '111221112',
        '5': '211221111',
        '6': '112221111',
        '7': '111211212',
        '8': '211211211',
        '9': '112211211',
        'A': '211112112',
        'B': '112112112',
        'C': '212112111',
        'D': '111122112',
        'E': '211122111',
        'F': '112122111',
        'G': '111112212',
        'H': '211112211',
        'I': '112112211',
        'J': '111122211',
        'K': '211111122',
        'L': '112111122',
        'M': '212111121',
        'N': '111121122',
        'O': '211121121',
        'P': '112121121',
        'Q': '111111222',
        'R': '211111221',
        'S': '112111221',
        'T': '111121221',
        'U': '221111112',
        'V': '122111112',
        'W': '222111111',
        'X': '121121112',
        'Y': '221121111',
        'Z': '122121111',
        '-': '121111212',
        '.': '221111211',
        ' ': '122111211',
        '$': '121212111',
        '/': '121211121',
        '+': '121112121',
        '%': '111212121',
    }

    start = stop = '121121211'

    lines = []
    lines.append(start + '1')
    for character in code:
        lineset = codeset.get(character, None)
        assert lineset, "Illegal character in Code 39: "+repr(character)
        lines.append(lineset + '1')
    lines.append(stop)

    return ''.join(lines)


def code_ean13(code, **kw):
    '''
    Perform EAN13 or EAN8 encoding.
    >>> code_ean13('5901234123457')
    '11131121123122221221411231111111222121221411113212311312111'
    >>> code_ean13('12345670')
    '1112221212214111132111111231111413123211111'
    '''

    # If the length is either 12 or 7, add a checksum digit
    if len(code) in (7, 12):
        code = code + str(ean13_checksum(code))

    code_choice = [
        'LLLLLL', 'LLGLGG', 'LLGGLG', 'LLGGGL', 'LGLLGG',
        'LGGLLG', 'LGGGLL', 'LGLGLG', 'LGLGGL', 'LGGLGL',
        ]

    r_code = [
        '1110010', '1100110', '1101100', '1000010', '1011100',
        '1001110', '1010000', '1000100', '1001000', '1110100',
        ]
    # L is complement of R, G is reverse of R

    def complement(b):
        out = ''
        for c in b:
            out += ('1' if c == '0' else '0')
        return out

    def reverse(b):
        return ''.join(reversed(b))

    assert code.isdigit(), "only digits allowed in EAN13 encoding"
    assert len(code) in (8, 13), "number of digits must be 8 or 13"

    start = '101'
    stop = '101'
    mid = '01010'

    if len(code) == 13:
        # Choice of codes for EAN-13
        first = int(code[0])
        left = code[1:7]
        right = code[7:13]
        codes = code_choice[first]

    else:
        # Codes for EAN-8
        left = code[0:4]
        right = code[4:8]
        codes = 'LLLL'

    bits = []
    bits.append(start)
    for i in range(len(codes)):
        b = r_code[int(left[i])]
        if codes[i] == 'L':
            b = complement(b)
        if codes[i] == 'G':
            b = reverse(b)
        bits.append(b)
    bits.append(mid)
    for i in range(len(codes)):
        b = r_code[int(right[i])]
        bits.append(b)
    bits.append(stop)

    bits = ''.join(bits)

    # Convert bits to lines
    lines = []
    next = '0'
    while bits:
        try:
            ind = bits.index(next)
        except ValueError:
            ind = len(bits)

        lines.append(str(ind))
        next = str(1 - int(next))
        bits = bits[ind:]

    return ''.join(lines)


def ean13_checksum(code):
    '''
    Calculate checksum for EAN-13
    >>> ean13_checksum('590123412345')
    '7'
    >>> ean13_checksum('1234567')
    '0'
    '''

    # Rules:
    # - rightmost digit (not including checksum) is odd,
    # - even numbers are weighted 1,
    # - odd numbers are weighted 3,
    # - checksum is included as last (even) component,
    # - result modulo 10 must be 0.

    assert code, "Code missing"

    code = str(code)
    assert code.isdigit(), "Code must only contain digits"

    checksum = -1
    weight = (len(code) % 2 == 0) and 1 or 3
    for digit in code:
        checksum += int(digit) * weight
        weight = 4 - weight
    checksum = 9 - (checksum % 10)

    return str(checksum)


def code_i2of5(code, **kw):
    '''Encode an even number of digits using interleaved 2 of 5.
    If you pass an odd number, this routine will fill up with a checksum.
    >>> code_i2of5('1236')
    '111121121111222122121111211'
    >>> code_i2of5('123')
    '111121121111222122121111211'
    '''

    codeset = [
        '11221', '21112', '12112', '22111', '11212',
        '21211', '12211', '11122', '21121', '12121',
    ]

    start = '1111'
    stop = '211'

    assert code.isdigit(), "only digits allowed in I2of5 encoding"
    if len(code) % 2 != 0:
        # add checksum
        code += str(i2of5_checksum(code))
    assert len(code) % 2 == 0, "use an even number of digits in I2of5"

    lines = []
    lines.append(start)
    while code:
        bars = codeset[int(code[0])]
        spaces = codeset[int(code[1])]
        code = code[2:]

        for i in range(5):
            lines.append(bars[i])
            lines.append(spaces[i])
    lines.append(stop)

    return ''.join(lines)


def ean128_code(code, plain=False, **kw):
    '''Use the checksum-enabled encoder to calculate the lineset for a code.
    EAN128 is default. For plain Code 128, set the "plain" flag.
    >>> ean128_code('(9901)23')
    '2112324111311131412221223121314212112331112'
    '''
    return code_128(code=code, plain=plain, **kw)


def code_128(code, plain=True, ai_list=None, **kw):
    '''Use the checksum-enabled encoder to calculate the lineset for a code.
    Plain Code 128 is default. For GS1-128, set "plain" to False.
    >>> code_128('990123')
    '2112321131412221223121311122142331112'
    >>> code_128('A1234')
    '2112141113231131411122321311231141132331112'
    '''

    codeset = [
        "212222", "222122", "222221", "121223", "121322",  # 00 - 04
        "131222", "122213", "122312", "132212", "221213",
        "221312", "231212", "112232", "122132", "122231",  # 10 - 14
        "113222", "123122", "123221", "223211", "221132",
        "221231", "213212", "223112", "312131", "311222",  # 20 - 24
        "321122", "321221", "312212", "322112", "322211",
        "212123", "212321", "232121", "111323", "131123",  # 30 - 34
        "131321", "112313", "132113", "132311", "211313",
        "231113", "231311", "112133", "112331", "132131",  # 40 - 44
        "113123", "113321", "133121", "313121", "211331",
        "231131", "213113", "213311", "213131", "311123",  # 50 - 54
        "311321", "331121", "312113", "312311", "332111",
        "314111", "221411", "431111", "111224", "111422",  # 60 - 64
        "121124", "121421", "141122", "141221", "112214",
        "112412", "122114", "122411", "142112", "142211",  # 70 - 74
        "241211", "221114", "413111", "241112", "134111",
        "111242", "121142", "121241", "114212", "124112",  # 80 - 84
        "124211", "411212", "421112", "421211", "212141",
        "214121", "412121", "111143", "111341", "131141",  # 90 - 94
        "114113", "114311", "411113", "411311", "113141",
        "114131", "311141", "411131", "211412", "211214",  # 100 - 104
        "211232", "2331112",
    ]

    if isinstance(code, list):
        # User has passed raw 128 codes
        codes = code
    else:
        codes = ean128_to_raw128(code, checksum=True, plain=plain, ais=ai_list)

    lines = [codeset[a] for a in codes]
    return ''.join(lines)


def ean128_split_hri(code, ais=None):
    '''Split a GS1 HRI into parts that need to start with FNC1 ('>F')
    >>> ean128_split_hri('(9901)012312(20)12(51)23123456')
    ['9901012312', '20125123123456']
    '''
    ais = ais or ai_list()
    constant_ais = [a for a in ais.keys() if ais[a]]

    parts = []
    part = ""
    entries = code.split('(')
    for entry in entries[1:]:
        ai, value = entry.split(')')
        part += ai+value

        # non-standard AIs require a FNC1 separator
        if ai not in constant_ais:
            parts.append(part)
            part = ""

    if part != "":
        # could, in theory, prepend the first part to possibly save a byte
        parts.append(part)

    return parts


def code128_group_data(data, starting_codeset=None):
    '''Turns data into a list of consecutive characters grouped by code set
    (only 'B' and 'C').
    >>> code128_group_data('12A345')
    [('C', '12'), ('B', 'A3'), ('C', '45')]
    '''
    # Page A not used here. (ASCII with no lower letters, unprintable control
    # chars instead)

    def count_digits(data):
        i = 0
        while i < len(data) and data[i].isdigit():
            i += 1

        return i

    # TODO right now starting_codeset changes nothing.
    # that behaviour was kept to produce barcodes consistent with the previous
    # code.
    # currently, the code set changes to 'C' for as few as two digits which
    # results in a one byte overhead for e.g. '(12)' (4 symbols in plain B,
    # 5 when going back and forth).

    if starting_codeset:
        enc = starting_codeset
    else:
        # assume code B (full ASCII), but if code starts with two digits,
        # change to C
        enc = 'B'
        if data[0][:2].isdigit():
            enc = 'C'

    assert enc in ['B', 'C']

    grouped = []

    while len(data) > 0:
        count = 0

        if enc == 'B':
            # collect non-digits straight away
            while count < len(data) and not data[count].isdigit():
                count += 1

            # TODO continue with 'B' unless there are at least 4 digits
            # see comment above

            # collect next digit only if odd number of digits follows
            digits = count_digits(data[count:])

            if (digits % 2) == 1:
                count += 1
        elif enc == 'C':
            # collect pairs of digits
            digits = count_digits(data)

            count = digits & ~1

        if count > 0:
            grouped.append((enc, data[:count]))
            data = data[count:]

        # TODO something more mergeable
        enc = chr(ord(enc) ^ 1)

    return grouped


def ean128_to_raw128(code, checksum=None, plain=None, ais=None):
    '''EAN 128 parser. Feed the string with ais in brackets and get a raw 128
    string.
    Set the flag "plain" to disable EAN AI parsing.
    >>> ean128_to_raw128('(9901)0123')
    '105 102 99 1 1 23'
    >>> ean128_to_raw128('(9901)1A33')
    '105 102 99 1 100 17 33 99 33'
    '''

    codeB = (''' !"#$%&`()*+'-./0123456789:;<=>?@ABCDEFGHIJKLMN'''
             '''OPQRSTUVWXYZ[\\] _`abcdefghijklmnopqrstuvwxyz{|}~'''
             '''\x7f\x1c''')

    if plain:
        parts = [code, ]
    else:
        parts = ean128_split_hri(code, ais)

    grouped_parts = []

    last_codeset = None
    for part in parts:
        grouped = code128_group_data(part, last_codeset)
        grouped_parts.append(grouped)

        last_codeset = grouped[0][0]

    output = []

    if len(grouped_parts) > 0:
        last_codeset = grouped_parts[0][0][0]

        # START_B, START_C
        output.append(last_codeset == 'B' and 104 or 105)

    for part in grouped_parts:
        if not plain:
            # FNC1
            output.append(102)

        for codeset, data in part:
            if codeset != last_codeset:
                # switch code set
                # might want to always do this except for the first segment
                output.append(codeset == 'B' and 100 or 99)

            if codeset == 'B':
                for char in data:
                    output.append(codeB.index(char))
            else:
                # for pair in zip(*[iter(data)] * 2):
                #     pair = ''.join(list(pair))
                for pair_index in range(0, len(data), 2):
                    pair = data[pair_index:pair_index + 2]
                    output.append(int(pair))

            last_codeset = codeset

    if not checksum:
        # Map into output readable by 'barcode -e 128raw'
        # TODO this shouldn't be here or 'checksum' is the wrong name
        string = ' '.join(map(str, output))
        return string

    # Calculate the checksum here and add stop code.
    chksum = output[0]
    chkcnt = 0
    for val in output:
        chksum += val * chkcnt
        chkcnt += 1
    chksum = chksum % 103
    output.append(chksum)
    output.append(106)  # STOP
    return output


def ai_list():
    '''Constant AI list useful for telling variable length AIs from
    constant length AIs.
    >>> len(ai_list().items())
    40
    '''
    ais = {
        '00': 18,
        '01': 14,
        '02': 14,
        '10': None,
        '11': 6,
        '13': 6,
        '15': 6,
        '17': 6,
        '20': 2,
        '21': None,
        '22': None,
        '23': None,
        '240': None,
        '250': None,
        '30': None,
        '37': None,
        '400': None,
        '410': 13,
        '411': 13,
        '412': 13,
        '414': 13,
        '420': None,
        '421': None,
        '8001': 14,
        '8002': None,
        '8003': None,
        '8005': 6,
        '8100': 6,
        '8101': 10,
        '8102': 2,
        '90': None,
        '91': None,
        '92': None,
        '93': None,
        '94': None,
        '95': None,
        '96': 123,
        '97': None,
        '98': None,
        '99': None,
    }
    return ais


def i2of5_checksum(code, **kw):
    '''Calculate checksum for i2of5 labels.
    >>> i2of5_checksum('123')
    6
    '''

    # Rules:
    # - even numbers are weighted 4,
    # - odd numbers are weighted 9,
    # - result modulo 10

    # Need first 12 digits of code
    assert code, "Code missing"
    assert code.isdigit(), "Code must only contain digits"

    checksum = -1
    weight = 4
    for digit in code:
        checksum += int(digit) * weight
        weight = 13 - weight
    checksum = 9 - (checksum % 10)

    return checksum


def tex_code(code='(9901)1234',
             heightmm=10, lwmm=0.25, offset=0.05,
             plain=None, coding='ean128', **kw):
    '''Generate TeX code for the given barcode.
    The option "plain" is unused and deprecated.

    >>> quickhash(tex_code(code='(9901)1234', heightmm=10,
    ...           lwmm=0.25, offset=0.05,
    ...           coding='ean128'))
    '8e2cdc14e72abf9af881dbbe3550280b'

    >>> quickhash(tex_code(code='99011234', heightmm=10,
    ...           lwmm=0.25, offset=0.05,
    ...           coding='code128'))
    '0aa6f22657815b9dbbe3ba294497964a'

    >>> int(len(tex_code(code='http://www.perfact.de', lwmm=0.25,
    ...     coding='qrcode')) / 100)
    122

    >>> quickhash(tex_code(code=b'http://www.perfact.de', lwmm=0.25,
    ...           coding='datamatrix'))  # doctest: +SKIP
    '270f694d972ea3af5d1bf761873df77e'
    '''

    modes = {
        'qrcode': '2d',
        'datamatrix': '2d',
    }
    mode = modes.get(coding, '1d')

    methods = {
        'ean128':     ean128_code,
        'gs1128':     ean128_code,
        'code128':    code_128,
        'i2of5':      code_i2of5,
        'ean13':      code_ean13,
        'ean8':       code_ean13,
        'code39':     code_39,
        'qrcode':     qrcode,
        'datamatrix': datamatrix,
    }
    method = methods.get(coding, None)
    assert method, "Unknown encoding: %s" % coding

    if mode == '1d':
        line = method(code, **kw)

        punch_fmt = '\\rule{%(plw).3fmm}{%(lh).3fmm}'
        break_fmt = '\\kern%(blw).3fmm'

        out = []
        brk = True
        for c in line:
            brk = not brk
            plw = int(c) * lwmm - offset
            blw = int(c) * lwmm + offset
            out.append((brk and break_fmt or punch_fmt) %
                       {'plw': plw, 'blw': blw, 'lh': heightmm})

    if mode == '2d':
        size, lines = method(code)

        strut = '\\rule{0mm}{%.3fmm}' % lwmm
        punch = '\\rule{%.3fmm}{%.3fmm}' % (lwmm, lwmm)
        blank = '\\kern%.3fmm' % lwmm

        out = []
        for line in lines.split('\n'):
            out.append('\\nointerlineskip\\vbox{'+strut)
            for c in line:
                out.append((c == ' ') and blank or punch)
            out.append('}\n')

    return ''.join(out)


def gs1_parse(code):
    """ Parse given gs1 code and replace the use of ais in brackets with the
    use of the seperator '\x1d'.

    The result always starts with the seperator. followed by the ais and their
    values.

    If the value of an ai is variable in length, the end is specified by
    the seperator '\x1d'.

    Unknown ais will be treated as of variable length.
    If the length of an ai does not match the predefined length, a ValueError
    is raised.

    Test if the content of logon cards is parsed correctly:
    >>> gs1_parse('(9901)1234(9900)5678')
    '\\x1d99011234\\x1d99005678'

    Variable length ais must be followed by a seperator:
    >>> gs1_parse('(10)1234(20)56')
    '\\x1d101234\\x1d2056'

    The seperator must not be used as a terminator:
    >>> gs1_parse('(20)12(10)3456')
    '\\x1d2012103456'

    Invalid length of an ai has to result in a ValueError:
    >>> gs1_parse('(00)1234(20)56')
    Traceback (most recent call last):
        ...
    ValueError: Illegal value length for ai "00". Expected length 18, got 4

    Sub-Ais of 99 must be recognized as variable length no matter where
    the ")" is placed.
    >>> gs1_parse('(9904)1234(20)56')
    '\\x1d99041234\\x1d2056'
    >>> gs1_parse('(99)041234(20)56')
    '\\x1d99041234\\x1d2056'
    """
    seperator = '\x1d'
    ais = ai_list()

    # always start with '\x1d'
    result = seperator
    # set to True if the last ai requires a seperator before appending the
    # next ai
    append_seperator = False
    for part in code.split('('):
        # skip empty parts. Should only be the first entry in split result
        if part == '' or (')' not in part):
            continue

        ai, value = part.split(')', 1)
        if append_seperator:
            result += seperator
            append_seperator = False
        # if the ai is unknown or its value is of variable length a seperator
        # has to be added before appending another ai
        if ai not in ais.keys() or ais[ai] is None:
            append_seperator = True
        # validate the length of the ais value if it is predetermined
        else:
            if len(value) != ais[ai]:
                error_msg = (
                    'Illegal value length for ai "{}". '
                    'Expected length {}, got {}'
                ).format(ai, ais[ai], len(value))
                raise ValueError(error_msg)
        result += ai + value
    return result
