#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# RestConnections.py  -  datasource connections for REST used in dbconnector.py
#
# $Revision: 2.10 $
#
# Copyright (C) 2016 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

import re
import requests
import perfact.webservice


class SessionConnection(object):
    ''' For REST calls with session cookie like in abas'''

    def __init__(self, connector, dbtype=''):
        self.connector = connector
        self.dbtype = dbtype
        # get the credentials from the url
        match = perfact.webservice.credentialsFromUrl(connector)
        self.credentials = {}
        if match:
            self.credentials['username'] = match[0]
            self.credentials['password'] = match[1]
        self.login_url = perfact.webservice.removeCredentialsFromUrl(
            self.connector)
        self.logout_url = None
        self.session = None
        self.connect()

    def connect(self):
        # logout when need to
        self.disconnect()
        # create a new session
        self.session = requests.Session()
        res = self.session.post(
            self.login_url,
            data={
                'j_username': self.credentials['username'],
                'j_password': self.credentials['password'],
            }
        )
        if res.status_code != requests.codes.ok:
            # we got no session and may exot here
            self.session = None
            return False
        # Parse out the logout-Link
        link = self.parseForLogout(res.text)
        if link:
            pos = self.login_url[8:].find('/') + 8
            server_and_port = self.login_url[:pos]
            self.logout_url = server_and_port + link
        else:
            self.logout_url = None
        return True

    def disconnect(self):
        if self.session and self.logout_url:
            self.session.post(self.logout_url)
        self.session = None

    def parseForLogout(self, html):
        ''' Try to guess a logout-link from html'''
        link = None
        m = re.match(r'<a\s+href="(.+logout.*)">.*</a>', html)
        if m:
            link = re.match.groups()[0]
        return link

    def __del__(self):
        self.disconnect()

    @property
    def connected(self):
        if self.session:
            return True
        else:
            return False


class RestPseudoConnection(object):
    ''' For REST calls without any state information '''

    def __init__(self, connector, dbtype='', authtype='basic'):
        self.connector = connector
        self.dbtype = dbtype
        self.authtype = authtype
        # get the credentials from the url
        creds = perfact.webservice.credentialsFromUrl(connector)
        self.credentials = None
        if creds:
            self.credentials = {}
            self.credentials['username'] = creds[0]
            self.credentials['password'] = creds[1]
        self.endpoint =\
            perfact.webservice.removeCredentialsFromUrl(self.connector)

    def close(self):
        pass
