#!/usr/bin/python
# coding: utf8
#
# AzureConnection.py  -  A wrapper for Microsoft's Azure-SDK
#
# $Revision: 1.10 $
#
# Copyright (C) 2023 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# $Id: dbconnector.py,v 2.10 2013/09/10 13:29:47 perfact Exp perfact $
from . import ProviderInterface
try:
    from azure.iot.device import IoTHubDeviceClient, ProxyOptions
    import socks
    import json
except ImportError:
    IoTHubDeviceClient = None


class Azure(ProviderInterface):
    def __init__(self, classname, sdk_connstring, proxy=None):
        super(ProviderInterface, self).__init__()
        if not IoTHubDeviceClient:
            raise Exception(
                'Required package python3-azure for Azure '
                'connection not installed')

        self.classname = classname
        self.sdk_connstring = sdk_connstring
        self.proxy_config = proxy or {}
        self.session = None

    def _should_use_proxy(self):
        """ Check if this AzureConnection should use a proxy based on given
        config variables
        """

        return (
            'addr' in self.proxy_config and
            'port' in self.proxy_config and
            'type' in self.proxy_config
        )

    def connect(self):
        if self.session:
            return

        if not self._should_use_proxy():
            self.session = IoTHubDeviceClient.create_from_connection_string(
                self.sdk_connstring,
            )
        else:
            proxy_options = ProxyOptions(
                proxy_type=getattr(socks, self.proxy_config['type']),
                proxy_addr=self.proxy_config['addr'],
                proxy_port=self.proxy_config['port'],
            )
            self.session = IoTHubDeviceClient.create_from_connection_string(
                self.sdk_connstring,
                proxy_options=proxy_options,
            )

        self.session.connect()

    def disconnect(self):
        if self.session:
            self.session.shutdown()
            self.session = None

    def close(self):
        self.disconnect()

    def read(self, query):
        """Looks like we can't really read from Azure
        But Azure can send messages to us which can be handled by setting
        client.on_message_received to a handler-function
        https://learn.microsoft.com/de-de/python/api/azure-iot-device/azure.iot.device.iothubdeviceclient?view=azure-python#azure-iot-device-iothubdeviceclient-receive-message
        """
        raise NotImplementedError()

    def write(self, data):
        self.connect()

        # Make sure the data has as format the SDK can understand
        if type(data) in [dict, list]:
            data = json.dumps(data)

        self.session.send_message(data)

    @property
    def connected(self):
        if not self.session:
            return False

        return getattr(self.session, "connected", False)
