#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# fwrc.py  -  Firewall remote control
#
# Copyright (C) 2016 Thorsten Südbrock <thorsten.suedbrock@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import os
import re
from .generic import safe_syscall


def fwrcmd(params, proxy="vpnnode1"):
    """Issue a firewall (a.k.a. netfilter) command on a remote system
    """
    cmd = 'ssh %s sudo /home/mpaproxy/netfilter/netfilter.py "%s"' \
          % (proxy, params)
    ret, out = safe_syscall(cmd, raisemode=True)
    assert ret == 0, 'Communication with netfilter module failed: %s' % out
    return out


def test_all():
    """Tests need to be run as user 'zope'!
    """
    uid = os.getuid()
    assert uid == 1005, "You need to run as user 'zope' (uid=1005)! \n\
    Your uid is %s. As root try: 'sudo -u zope python fwrc.py'" % uid

    ruledict = {'action': 'insert',
                'table': 'nat',
                'chain': 'POSTROUTING',
                'src': '10.255.255.1/32',
                'proto': 'tcp',
                'out': 'eth0',
                'target': 'LOG',
                'target-params': {'log-prefix': 'test', 'log-level': '3'},
                'matches': [('tcp', {'sport': '4711', 'dport': '1337'}),
                            ('comment', {
                             'comment': '0123-Testdurchlauf des Moduls'})
                            ]
                }

    print('Inserting firewall rule')
    res = fwrcmd(ruledict)
    assert res, 'Rule could not be inserted!'

    print('Deleting firewall rule')
    ruledict['action'] = 'delete'
    res = fwrcmd(ruledict)
    assert res, 'Rule could not be deleted!'

    print('All tests successful')


if __name__ == '__main__':
    test_all()


#
# This section taken from Mohn Media (only customer with live installation)
#

# Addition for generating vnc passwords for windows clients

def vncpasswd(password):
    ch_in, ch_out = os.popen2(
        'tightvncpasswd -f | hexdump -v -e \'/1 "%02x"\'')
    ch_in.write(password)
    ch_in.close()
    ret = ch_out.read().strip()
    ch_out.close()
    return ret

# Addition for managing GENUBOX connections
# Explanation of some values:
# vpnproxy = alias in the /root/.ssh/config file


def genuboxOpenConnection(ovpnconn_id, genubox_ipaddr, genubox_remote=None):
    # We have to strip the trailing VLSM/CIDR notation /xy if it is there
    genubox_ipaddr = re.sub(r'/([0-9]|[12][0-9]|3[0-2])$', '', genubox_ipaddr)
    if genubox_remote:
        genubox_remote = re.sub(
            r'/([0-9]|[12][0-9]|3[0-2])$', '', genubox_remote)
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/genuboxcommand --cmdline=open '
        '--ovpnconn_id=%s --genubox_ipaddr=%s --genubox_remote=%s' %
        (ovpnconn_id, genubox_ipaddr, genubox_remote or ''), 'r'
    )
    answer = fh.read()
    fh.close()
    # let the calling zope script decide what to do - we have to tear down the
    # firewall and so on..
    return answer


def genuboxCloseConnection(ovpnconn_id, genubox_ipaddr, genubox_remote=None):
    # We have to strip the trailing VLSM/CIDR notation /xy if it is there
    genubox_ipaddr = re.sub(r'/([0-9]|[12][0-9]|3[0-2])$', '', genubox_ipaddr)
    if genubox_remote:
        genubox_remote = re.sub(
            r'/([0-9]|[12][0-9]|3[0-2])$', '', genubox_remote)
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/genuboxcommand --cmdline=close '
        '--ovpnconn_id=%s --genubox_ipaddr=%s --genubox_remote=%s' %
        (ovpnconn_id, genubox_ipaddr, genubox_remote or ''), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution. Returned code "+str(ret)
        )
    return answer


def genuboxConnectionStatus(genubox_ipaddr):
    # We have to strip the trailing VLSM/CIDR notation /xy if it is there
    genubox_ipaddr = re.sub(r'/([0-9]|[12][0-9]|3[0-2])$', '', genubox_ipaddr)
    fh = os.popen(
        'ssh vpnproxy /home/mpaproxy/genuboxcommand --cmdline=status '
        '--genubox_ipaddr=%s' % genubox_ipaddr, 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer

# Addition for generic firewall management (GENUBOX, MB-Router,
# TelePresence... - you name it)
#  We need root privileges on the vpnproxy system so we have to use 'sudo'
#  The script on vpnproxy must be executable and the /etc/sudoers file has to
#  be setup correctly
#  to allow the user 'mpaproxy' the execution without prompting for a password

# Create a new chain for the connections to/from the external host


def firewallNew(ovpnconn_id, tag, targetipaddr):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=new '
        '--ovpnconn_id=%d --tag=%s --snaconn_targetipaddr=%s' %
        (ovpnconn_id, tag, targetipaddr), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer


# Add a new rule in the chain to allow the specified IP, Port, Protocol and
# the direction (inbound/outbound)
def firewallCreate(ovpnconn_id, tag, targetipaddr, ovpntarget_ipaddr,
                   ovpntarget_portdef, ovpntarget_proto, direction,
                   ovpntarget_id):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=create '
        '--ovpnconn_id=%d --tag=%s --snaconn_targetipaddr=%s '
        '--ovpntarget_ipaddr=%s --ovpntarget_portdef=%s --ovpntarget_proto=%s '
        '--direction=%s --ovpntarget_id=%d' %
        (ovpnconn_id, tag, targetipaddr, ovpntarget_ipaddr, ovpntarget_portdef,
         ovpntarget_proto, direction, ovpntarget_id), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer


# List all Rules with the specified tag. ovpnconn_id is just a dummy value
# in this function and could be set to any value!
def firewallListChain(tag):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=list '
        '--tag=%s' % (tag), 'r'
    )
    answer = fh.read()
    answer = answer.split('\n')
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer


# Gather the traffic for the session
def firewallCheck(ovpnconn_id, tag):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=check '
        '--ovpnconn_id=%d --tag=%s' % (ovpnconn_id, tag), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer


# Remove ALL rules and the chain afterwards
def firewallDisconnect(ovpnconn_id, tag):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=disconnect '
        '--ovpnconn_id=%d --tag=%s' % (ovpnconn_id, tag), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer


# Add a new rule to DNAT packets matching the NAT_DST_IP and PORT parameters
# to redirect them to REAL_DST_IP
# If the NAT_DST_IP is set to 'None' the official ip-address of the vpn-server
# will be used
# If the NAT_SRC_IP is set to None the ip-address 0.0.0.0/0 will be used,
# matching ANY network
def firewallCreateDnat(ovpnconn_id, tag, nat_dst_ip, ovpntarget_ipaddr,
                       ovpntarget_portdef, ovpntarget_proto, nat_src_ip=None):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=dnat-on '
        '--ovpnconn_id=%d --tag=%s --snaconn_targetipaddr=%s '
        '--ovpntarget_ipaddr=%s --ovpntarget_portdef=%s --ovpntarget_proto=%s '
        '--nat_src_ip=%s' %
        (ovpnconn_id, tag, nat_dst_ip, ovpntarget_ipaddr, ovpntarget_portdef,
         ovpntarget_proto, nat_src_ip or ''), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer


# Drop the defined DNAT roules
def firewallDropDnat(ovpnconn_id, tag):
    fh = os.popen(
        'ssh vpnproxy sudo /home/mpaproxy/fwcommand --cmdline=dnat-off '
        '--ovpnconn_id=%d --tag=%s' % (ovpnconn_id, tag), 'r'
    )
    answer = fh.read()
    ret = fh.close()
    if ret is not None:
        raise ValueError(
            "Error in remote script-execution returned code "+str(ret))
    return answer
