#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# loggongtools.py  -  helper for logging in daemons and command line programs
#
# $Revision: 1.0 $
#
# Copyright (C) 2018 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#

''' Example in bin/perfact-loggingexample '''

import logging


def addArgs(parser, name):
    ''' Enrich the given ArgumentParser with Arguments for logging.

    parser -- ArgumentParser that gets additional arguments and is returned.
    name -- Name of the application using the logger. Used for logfile name.
    '''
    # use this or just copy the code
    parser.add_argument('--log-to', '-l', type=str, default=target_console,
                        help='Logging output: journal, file or console '
                        '(defaults to %s)' % target_console)
    parser.add_argument('--log-file', '-f', type=str,
                        default="/var/log/%s.log" % name,
                        help='Log file (defaults to '
                        '/var/log/%s.log).' % name)
    parser.add_argument('--debug', '-d', action="store_const",
                        default=logging.INFO, const=logging.DEBUG,
                        help='Log debugging information (default: no)')


def configureLogger(args, logger):
    ''' Configure the given logger with the given args.

    args -- Arguments from ArgumentParser.parse_args or a loggingArgs instance.
    logger -- logger to configure.
    '''
    assert args.log_to in target2handler, "invalid log target set."
    handler = target2handler[args.log_to](args)
    formatter = logging.Formatter(
        target2format[args.log_to],
        "%Y-%m-%dT%H:%M:%S")
    handler.setFormatter(formatter)
    logger.addHandler(handler)
    logger.setLevel(args.debug)


def createLogger(args, name):
    ''' Create a new logger from arguments and application name

    args -- Arguments from ArgumentParser.parse_args or a loggingArgs instance.
    name -- Name of the application using the logger. Used for logfile name.
    '''
    logger = logging.getLogger(name)
    configureLogger(args=args, logger=logger)
    return logger


def createConsoleHandler(args):
    ''' create a logging handler for console output '''
    return logging.StreamHandler()


def createFileHandler(args):
    ''' create a logging handler for file output '''
    import logging.handlers
    return logging.handlers.WatchedFileHandler(args.log_file)


def createJournalHandler(args):
    ''' create a logging handler for journal output '''
    import systemd.journal
    return systemd.journal.JournalHandler()


class LoggingArgs(object):
    ''' provide arguments for configureLogger
        if you do not have a Namespace Object from argparse '''

    def __init__(self, log_to='console', log_file='', log_level=logging.INFO):
        self.log_to = log_to
        self.log_file = log_file
        self.debug = log_level


# valid targets
target_console = 'console'
target_file = 'file'
target_journal = 'journal'

# log_formats
file_format = "%(asctime)s.%(msecs)03d %(name)s %(levelname)s %(message)s"
journal_format = "%(levelname)s %(message)s"
target2format = {
    target_console: file_format,
    target_file: file_format,
    target_journal: journal_format,
}

# log_handlers
target2handler = {
    target_console: createConsoleHandler,
    target_file: createFileHandler,
    target_journal: createJournalHandler,
}
