# -*- coding: utf-8 -*-
#
# mail.py  -  Mail related utilities
#
# Copyright (C) 2016 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

# For reading and processing emails
import sys
import email
import email.header
import email.utils
import shutil
import os

# For deploying mail meta data
from .dbconn import dbconn
from .fileassets import fileassets

# For uploading files
from .file import file_upload

# For converting to JSON
from .generic import json_encode
from .generic import html_quote, html_cleanup
from .generic import to_string

# For de-/encoding base64
from .generic import base64encode

# For logging
from . import say

# For input validation
import re

# Execution of arbitrary commands
from .generic import safe_syscall
from .generic import validate_domain
from .generic import validate_domain_list
from .generic import validate_port
from .generic import same_type

import six
if six.PY2:
    bytes = str
else:
    unicode = str


# XXX TODO: All doctests involving real IO and the use of "postfix" tools
# have been disabled!
# Use dryrun options or mocking to run these tests safely.


def binwrapper():
    '''Wrapper used by /usr/bin/perfact-mailreceiver
    Installation:
    - Check if /usr/bin/perfact-mailreceiver exists and is executable
    - Edit /etc/aliases to forward special addresses to zope
      e.g. service-address: zope
    - Check mydestination in /etc/postfix/main.cf and usage of /etc/aliases
       alias_maps = hash:/etc/aliases
       alias_database = hash:/etc/aliases
    - Modify /home/zope/.forward (|"/usr/bin/perfact-mailreceiver -") and check
      file properties (owner zope:zope, permissions 664)
    - postalias /etc/aliases
      (Postfix rereads the new /etc/aliases.db automatically)
    '''
    say.say('mailreceiver: Mail received at perfact.mail')
    # Make sure to read bytes
    if six.PY2:
        data = sys.stdin.read()
    else:
        data = sys.stdin.buffer.read()
    mail_receive(data)
    say.say('mailreceiver: Mail passed to perfact.mail.mail_receive')


def mail_decodeheader(value):
    """
    Decode a email header, returning a utf-8 encoded string.
    """
    out = ''
    for part, encoding in email.header.decode_header(to_string(value)):
        # If encoding is set, the part is given as bytes, else it is
        # usually a string, but sometimes it seems to still give bytes,
        # even in Python3! (can be observed if a header contains multiple
        # parts)
        out += to_string(part.decode(encoding) if encoding else part)
    return out.strip()


def mail_decodeheaders(msg):
    '''Decode headers, returning a string dictionary.
    Input:
    - msg: an email message object

    Environment: None

    Output:
    - A dictionary with header keys and string content.

    Side Effects: None
    '''
    return {
        key: mail_decodeheader(value)
        for key, value in msg.items()
    }


def mail_receive(infile,
                 username=None,
                 additional_bcc_headers=None, commit=True):
    '''Receive a message, either as bytes or as a file-like object, and
    pass it on to the database.

    Inputs:
    - infile: bytes or file-like object containing the message.
    - username: optional, defaults to __system__
    - additional_bcc_headers: optional list to use multiple header fields
      to determine the value for the 'bcc' field in the imported message.
      The first found value will be taken:
      e.g.: ['Bcc', 'Delivered-To', 'X-Original-To']

    Environment:
    - needs the database tables appemail* and file*

    Outputs:
    - The database appemail_id which the email received.

    Side Effects:
    - Puts attachments into the file repository, file and email
      metadata into the database.
    '''

    if additional_bcc_headers is None:
        additional_bcc_headers = ['Delivered-To', 'X-Original-To']

    # Read the message

    if isinstance(infile, str):
        msg = email.message_from_string(infile)
    elif isinstance(infile, bytes):
        msg = email.message_from_bytes(infile)
    else:
        msg = email.message_from_file(infile)

    headers = mail_decodeheaders(msg)

    # JSON version of headers
    headers_json = json_encode(headers)

    #  Get senders appusername
    if username is None:
        dbconn.execute(fileassets['mail.appemail_appusername'],
                       sender=email.utils.parseaddr(headers.get('From'))[1])
        resp = dbconn.tuples()
        if len(resp) > 0:
            username = resp[0][0]

    if username is None:
        username = '__system__'

    bcc = headers.get('Bcc')
    if bcc is None and isinstance(additional_bcc_headers, list):
        for header in additional_bcc_headers:
            bcc = headers.get(header)
            if bcc:
                break

    dbconn.execute(
        fileassets['mail.appemail_insert'],
        username=username,
        sender=headers.get('From'),
        receiver=headers.get('To'),
        subject=headers.get('Subject'),
        cc=headers.get('Cc'),
        bcc=bcc,
        headers=headers_json,
    )
    appemail_id = dbconn.tuples()[0][0]

    # Storage for the message body (We store both kinds)
    body_html = ''
    body_text = ''
    inline_images = {}  # dict Content-ID -> (content_type, payload)

    for part in msg.walk():

        content_type = part.get_content_type()
        filename = part.get_filename()
        payload = part.get_payload(decode=True)
        charset = part.get_content_charset()
        content_id = part.get('Content-ID')
        if charset is None:
            # charset can be None
            charset = 'utf-8'

        if payload is None:
            # Usually a multipart section, but might also be a broken part of
            # the email. In any case, we can't extract anything useful.
            continue

        if not filename:
            # Inline part. Could be the body.
            new_body = unicode(payload, charset, 'replace')

            if content_type == 'text/html' and not body_html:
                body_html = new_body
            if content_type == 'text/plain' and not body_text:
                body_text = new_body

            continue

        # Filename might be in encoded header format
        filename = mail_decodeheader(filename)

        file_id = file_upload(payload, filename=filename,
                              mimetype=content_type,
                              username=username, commit=False
                              )
        dbconn.execute(fileassets['mail.appemailxfile_insert'],
                       username=username,
                       appemail_id=appemail_id,
                       file_id=file_id)

        if (content_type in ['image/png', 'image/jpeg']
                and content_id is not None
                and len(content_id) > 0
                and content_id[0] == '<'
                and content_id[-1] == '>'):
            inline_images[content_id[1:-1]] = (
                (content_type, to_string(base64encode(payload).strip()))
            )

    # Build a clean html body

    def src_replacer(val):
        ''' replace img src tags by inline content if found in attachment. '''
        if val.startswith('cid:') and val[4:] in inline_images:
            return 'data:%s;base64,%s' % inline_images[val[4:]]
        else:
            return 'INVALID'

    if body_html:
        body = html_cleanup(body_html, custom_tags={
                            'img': [('src', src_replacer), ]})
    elif body_text:
        body = '<pre>' + html_quote(body_text) + '</pre>'
    else:
        body = new_body

    dbconn.execute(fileassets['mail.appemail_body'],
                   id=appemail_id,
                   body=body)

    # check if this database supports appemail lifecycles and update them
    dbconn.execute(fileassets['mail.appemaillc_exists'])
    resp = dbconn.tuples()
    if len(resp) > 0:
        dbconn.execute(fileassets['mail.appemail_lc'],
                       id=appemail_id)

    if commit:
        dbconn.commit()

    return appemail_id


def mail_encode_header(value, encoding='utf-8'):
    '''Return encoded version of the value.  Input is utf-8-encoded
    8bit or universal string. Output is ascii.

    >>> mail_encode_header('TestÄÖÜ')
    '=?utf-8?b?VGVzdMOEw5bDnA==?='
    '''

    # Make sure we have a unicode string
    if isinstance(value, bytes):
        value = value.decode(encoding)

    return email.header.Header(value).encode()


def validate_tls_protocols(inp):
    """Check if we have a valid protocol list, separated by whitespace.
    Negation with prepended ! is allowed.

    >>> validate_tls_protocols('!SSLv2, !SSLv3, TLSv1')
    True

    >>> validate_tls_protocols('SSLv4')
    False

    >>> validate_tls_protocols('SSLv2, TLSV8, !SSLv3')
    False
    """
    protocols = ['SSLv2', 'SSLv3', 'TLSv1', 'TLSv1.1', 'TLSv1.2']
    res = inp.split(',')
    count = len(res)
    valid = 0
    for item in res:
        item = item.strip()
        if item[0] == '!':
            item = item[1:]
        if item in protocols:
            valid += 1
    if valid == count:
        return True
    return False


def validate_relayhost(value):
    """Check for allowed values to be used with postfix directive
    relayhost = ...
    Returns True or False

    >>> validate_relayhost('server.example.com')
    True
    >>> validate_relayhost('server.example.com:123')
    True
    >>> validate_relayhost('server.example.com:123:456')
    False
    >>> validate_relayhost('localhost')
    True
    >>> validate_relayhost('127.0.0.1:25')
    True
    >>> validate_relayhost('10.1.2.3,127.0.0.1:25')
    False
    >>> validate_relayhost('10.1.2.3/24:25')
    False
    """
    res = value.split(':')
    if len(res) == 2:
        host, port = res
        if validate_domain(host):
            try:
                validate_port(port)
            except ValueError:
                return False
            return True
    elif len(res) == 1:
        host = res[0]
        return validate_domain(host)
    return False


def validate_regexp(value, pattern, flags=re.IGNORECASE):
    """Simple wrapper to check if a given value matches a given
    regular expression pattern
    Returns True or False
    """
    matched = False
    if flags:
        matched = re.match(pattern, value, flags) or False
    else:
        matched = re.match(pattern, value) or False
    return matched


def validate_maps(value):
    """Validate if a mapping file for postfix exists
    hash:/my/file/path, dbm:/another/file/path

    WARNING: Whitespaces in filenames are not supported yet!

    >>> setup = doctests_setup()  # doctest: +SKIP
    >>> validate_maps('hash:'+setup['mappingfile'])  # doctest: +SKIP
    True

    >>> validate_maps('hash:/non/existent')  # doctest: +SKIP
    False

    >>> validate_maps('hash:' + setup['mappingfile'] +
    ...     ',btree:' + setup['mappingfile'])  # doctest: +SKIP
    True

    >>> validate_maps('hash:' + setup['mappingfile'] +
    ...     ',      btree:' + setup['mappingfile'])  # doctest: +SKIP
    True
    >>> doctests_teardown(setup)  # doctest: +SKIP
    """
    pairs = value.split(',')
    validmaps = 0
    validpaths = 0
    for pair in pairs:
        res = pair.split(':', 1)
        if not len(res) == 2:
            return False
        maptype, filepath = res
        # valid mapping types: 'postconf -m'
        if maptype.strip() in ['hash', 'dbm', 'btree', 'texthash']:
            validmaps += 1
        if os.path.exists(filepath):
            validpaths += 1
    if validmaps == validpaths == len(pairs):
        return True
    return False


def postalias_create(path, maptype, raisemode=False, confdir=None):
    """Start conversion from a mapping template file (plain text) to a
    given mapping type using the 'postalias' command.
    Postalias has a syntax of:
      key:value
    OR
      key: value, value2, value3...

    >>> setup = doctests_setup()  # doctest: +SKIP
    >>> postalias_create(setup['mappingfile'], 'hash', raisemode=True,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    True

    >>> postalias_create(setup['mappingfile']+'_new', 'hash', raisemode=False,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    True

    >>> postalias_create(setup['mappingfile'], 'foobar', raisemode=False,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    False
    >>> doctests_teardown(setup)  # doctest: +SKIP
    """
    # create file if it does not exit
    if not os.path.exists(path):
        fd = open(path, 'w')
        fd.close()
    if confdir:
        errlvl, out = safe_syscall(['/usr/bin/sudo', '/usr/sbin/postalias',
                                    '-c', confdir, maptype+':'+path],
                                   raisemode=raisemode)
    else:
        errlvl, out = safe_syscall(['/usr/bin/sudo', '/usr/sbin/postalias',
                                    maptype+':'+path], raisemode=raisemode)

    if errlvl == 0:
        return True

    msg = ('ERROR while generating postfix alias map for "%s" with type "%s"'
           'Errorlevel: %s\nOutput: %s' % (path, maptype, errlvl, out))
    say.say(msg)
    return False


def postalias_update(path, key, value, raisemode=False, confdir=None):
    """Update an alias mapping file previously created by 'postalias'
    in place. This overrides values for already existing keys!

    >>> postalias_update(aliasmap, 'foo.bar.batz', 'testuser:secret',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    True

    >>> postalias_update('/nonexistent/file', 'invalid', 'nono',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    False

    >>> postalias_update(aliasmap, 'foo@bar.batz',
    ...     'user1, user2')  # doctest: +SKIP
    True

    >>> postalias_update(aliasmap, 'foo@bar',
    ...     'invalid value')  # doctest: +SKIP
    False
    """
    # input validation
    if len(key.split()) > 1:
        msg = 'ERROR - key "%s" contains whitespaces' % key
        say.say(msg)
        return False
    values = value.split(',')
    for value in values:
        val = value.strip()
        if not re.match(r'^[^ ]+$', val):
            return False

    if confdir:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/usr/sbin/postalias',
                '-i', '-r', '-o', '-c', confdir, path,
            ],
            stdin_data=key+':'+value, raisemode=raisemode,
        )
    else:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/usr/sbin/postalias',
                '-i', '-r', '-o', path,
            ],
            stdin_data=key+':'+value, raisemode=raisemode,
        )

    if errlvl == 0:
        return True
    msg = ('ERROR while updating postfix alias map at "%s" using key "%s"'
           'Errorlevel: %s\nOutput: %s' % (path, key, errlvl, out))
    say.say(msg)
    return False


def postalias_get(path, maptype='hash', raisemode=False, key=None,
                  confdir=None):
    """Query a postfix alias file for values. If a key is given, it only
    returns the value for that key. If key is unset, all key/value pairs in
    the mapping file are returned.
    A return value of None means the key/value pair is unset or the mapping
    file is empty.

    >>> postalias_get(btree_aliasfile, key='testkey', maptype='btree',
    ...     raisemode=True, confdir=setup['tmpdir'])  # doctest: +SKIP
    {'testkey': 'testvalue'}

    >>> postalias_get(btree_aliasfile, maptype='btree', raisemode=True,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'testkey2': 'testvalue2', 'testkey': 'testvalue'}

    >>> postalias_get('nonexistent', key='unknown',
    ...     raisemode=False)  # doctest: +SKIP
    False
    """
    if confdir:
        if key:
            # read out specific value
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postalias',
                    '-c', confdir, '-q', key,
                    maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
        else:
            # read out all values
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postalias',
                    '-c', confdir, '-s',
                    maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
    else:
        if key:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postalias',
                    '-q', key, maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
        else:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postalias',
                    '-s', maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
    if errlvl == 0:
        if key:
            return {key: out.strip()}
        else:
            ret = {}
            lines = out.split('\n')
            for line in lines:
                # drop some unwanted information
                if not len(line):
                    continue
                if line.startswith('@:'):
                    continue
                if line.startswith('YP_LAST_MODIFIED:'):
                    continue
                if line.startswith('YP_MASTER_NAME:'):
                    continue
                # add the interesting results to the ret dictionary
                key, value = line.split('\t')
                key = key.strip(':')
                ret[key] = value.strip()
            return ret
    msg = ('ERROR while querying postfix map at "%s" using key "%s"'
           'Errorlevel: %s\nOutput: %s' % (path, key, errlvl, out))
    say.say(msg)
    return False
    pass


def postmap_create(path, maptype='hash', raisemode=False, confdir=None):
    """Start conversion from a mapping template file (plain text) to a
    given mapping type using the 'postmap' command.
    Postmap files have a syntax of: key _space_ value

    >>> postmap_create(setup['mappingfile'], 'hash', raisemode=True,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    True

    >>> postmap_create(setup['mappingfile']+'_new', 'btree', raisemode=False,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    True

    >>> postmap_create(setup['mappingfile'], 'foobar', raisemode=False,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    False
    """
    # create source file if it does not exit - this prevents an error
    if not os.path.exists(path):
        fd = open(path, 'w')
        fd.close()
    if confdir:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/usr/sbin/postmap',
                '-c', confdir, maptype+':'+path,
            ],
            raisemode=raisemode,
        )
    else:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/usr/sbin/postmap',
                maptype+':'+path,
            ],
            raisemode=raisemode,
        )

    if errlvl == 0:
        return True

    msg = ('ERROR while generating postfix map for "%s" with type "%s"'
           'Errorlevel: %s\nOutput: %s' % (path, maptype, errlvl, out))
    say.say(msg)
    return False


def postmap_update(path, key, value, maptype='hash',
                   raisemode=False, confdir=None):
    """Update a mapping file previously created by 'postmap' in place.
    This overrides values for already existing keys!
    Note that the source file (plaintext) will not be altered.

    >>> postmap_update(passwordfile, 'foo.bar.batz', 'testuser:secret',
    ...     confdir=setup['tmpdir'], raisemode=True,
    ...     maptype='btree')  # doctest: +SKIP
    True

    >>> postmap_update('/nonexistent/file', 'invalid', 'nono',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    False
    """
    if confdir:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/usr/sbin/postmap',
                '-i', '-r', '-o', '-c', confdir, path,
            ],
            stdin_data=key+' '+value, raisemode=raisemode,
        )
    else:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/usr/sbin/postmap',
                '-i', '-r', '-o', path,
            ],
            stdin_data=key+' '+value, raisemode=raisemode,
        )

    if errlvl == 0:
        return True
    msg = ('ERROR while updating postfix map at "%s" using key "%s"'
           'Errorlevel: %s\nOutput: %s' % (path, key, errlvl, out))
    say.say(msg)
    return False


def postmap_get(path, maptype='hash', raisemode=False, key=None, confdir=None):
    """Query a postfix mapping file for values. If a key is given, it only
    returns the value for that key. If key is unset, all key/value pairs in
    the mapping file are returned.
    A return value of None means the key/value pair is unset or the mapping
    file is empty.

    >>> postmap_get(btree_mappingfile, key='testkey', maptype='btree',
    ...     raisemode=True, confdir=setup['tmpdir'])  # doctest: +SKIP
    {'testkey': 'testvalue'}

    >>> postmap_get(btree_mappingfile, maptype='btree', raisemode=True,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'testkey2': 'testvalue2', 'testkey': 'testvalue'}

    >>> postmap_get('nonexistent', key='unknown',
    ...     raisemode=False)  # doctest: +SKIP
    False
    """
    if confdir:
        if key:
            # read out specific value
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postmap',
                    '-c', confdir, '-q', key,
                    maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
        else:
            # read out all values
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postmap',
                    '-c', confdir, '-s',
                    maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
    else:
        if key:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postmap',
                    '-q', key, maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
        else:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postmap',
                    '-s', maptype + ':' + path,
                ],
                raisemode=raisemode,
            )
    if errlvl == 0:
        if key:
            return {key: out.strip()}
        else:
            ret = {}
            lines = out.split('\n')
            for line in lines:
                if not len(line):
                    continue
                key, value = line.split('\t')
                ret[key] = value
            return ret
    msg = ('ERROR while querying postfix map at "%s" using key "%s"'
           'Errorlevel: %s\nOutput: %s' % (path, key, errlvl, out))
    say.say(msg)
    return False


def whitelist_validation(key, value):
    """Whitelist for allowed main.cf keywords with additional value checking

    Checks are functions which get the value as first parameter and
    additional parameters depending on the function.

    >>> whitelist_validation('smtp_sasl_mechanism_filter', 'login, plain')
    True

    >>> whitelist_validation('relayhost', 'foo.example.com:465:123')
    False
    """
    global whitelist

    if key not in whitelist:
        return False

    if same_type(whitelist.get(key), []):
        func = whitelist[key][0]
        args = whitelist[key][1:]
        if func(value, *args):
            return True
    elif callable(whitelist.get(key)):
        func = whitelist[key]
        if func(value):
            return True
    return False


def postconf_main(key, value, raisemode=True, confdir=None):
    """Manipulate the /etc/postfix/main.cf configuration file using the
    builtin 'postconf' utility.
    Only whitelisted keys and matching RegExp for this keys are allowed.

    >>> postconf_main('nonexistent_key', 'unittest-foobar',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'new_value': 'unittest-foobar', 'message': 'Value "unittest-foobar" not\
 allowed for parameter "nonexistent_key" or key not in whitelist',\
 'old_value': None, 'key': 'nonexistent_key'}

    >>> postconf_main('mydestination', 'ema-devel-2016.perfact.de, localhost',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'new_value': 'ema-devel-2016.perfact.de, localhost', 'message':\
 'Nothing changed, old value "ema-devel-2016.perfact.de, localhost" is the\
 same as new value "ema-devel-2016.perfact.de, localhost"', 'old_value':\
 'ema-devel-2016.perfact.de, localhost', 'key': 'mydestination'}

    >>> postconf_main('smtp_sasl_auth_enable', 'yes',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'new_value': 'yes', 'message': None, 'old_value': 'no', 'key':\
 'smtp_sasl_auth_enable'}

    >>> postconf_main('relayhost', 'foo.example.com:465',
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'new_value': 'foo.example.com:465', 'message': None, 'old_value':\
 'smtp.perfact.de', 'key': 'relayhost'}

    >>> postconf_main('smtp_sasl_password_maps', 'hash:'+setup['mappingfile'],
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    {'new_value': 'hash:/tmp/python-modules_unittest_mail/mappingfile',\
 'message': None, 'old_value': '', 'key': 'smtp_sasl_password_maps'}
    """
    # a value of None is possible and will delete the key
    # (results in using the default)
    # the default is to modify
    delete = False
    if value is None:
        # delete the key
        delete = True
    else:
        # we treat all values as lowercase strings without whitespaces
        value = str(value)
        value = ' '.join(value.split())
        if not len(value):
            delete = True

    # default return value
    ret = {
        'key': key,
        'new_value': value,
        'old_value': None,
        'message': None,
    }

    # we want to change the value - check whitelisted values first!
    if not delete:
        if not whitelist_validation(key, value):
            msg = ('Value "%s" not allowed for parameter "%s"'
                   ' or key not in whitelist' % (value, key))
            ret.update(message=msg)
            return ret

    # backup old config
    errlvl, out, key, old_value = postconf_get(
        key=key,
        confdir=confdir,
        raisemode=raisemode,
    )
    ret.update(old_value=old_value)

    # sanitize the current configuration value
    if old_value is not None:
        cmp_value = old_value.lower()
        cmp_value = ' '.join(cmp_value.split())
        if cmp_value == '':
            cmp_value = None

    # if nothing changed we are finished
    if value == cmp_value:
        msg = ('Nothing changed, old value "%s" is the same as new '
               'value "%s"' % (cmp_value, value))
        ret.update(message=msg)
        return ret

    if delete:
        # delete the key, so the default is used again
        postconf_set(key, value=None, confdir=confdir, raisemode=raisemode)
    else:
        # modify the key, so the new value is used
        # change value - sudo needed!
        postconf_set(key, value, confdir=confdir, raisemode=raisemode)
        res = postconf_get(key, confdir=confdir, raisemode=raisemode)
        new_value = res[3]
        if new_value != value:
            msg = 'New value for key "%s" was not set! %s' % (
                new_value, str(res)
            )
            ret.update(message=msg)
            return ret

    # try to reload postfix with the new values
    errlvl, output = postfix_daemon(cmd='reload', raisemode=raisemode)

    if errlvl != 0:
        postconf_set(key, old_value, confdir=confdir, raisemode=raisemode)
        postfix_daemon(cmd='reload', raisemode=raisemode)
        msg = 'Restored old configuration "%s = %s"' % (key, old_value)
        ret.update(message=msg)

    return ret


def postconf_set(key, value=None, raisemode=False, confdir=None):
    """Set a configuration directive using the 'postconf' utility
    to the given value

    IMPORTANT: enable /usr/sbin/postconf in /etc/sudoers to be executable
    via 'sudo' without entering a password for the user of this function
    (e.g. zope)

    zope ALL=NOPASSWD:/usr/sbin/postconf

    >>> postconf_set('inet_interfaces', 'all', raisemode=True,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    (0, '', 'inet_interfaces', 'all')

    >>> postconf_set('nonexistent_key', 'unittest',
    ...     confdir=setup['tmpdir'])[0:3]  # doctest: +SKIP
    (0, '', 'nonexistent_key')

    >>> postconf_set('nonexistent_key', raisemode=False,
    ...     confdir=setup['tmpdir'])  # doctest: +SKIP
    (0, '', 'nonexistent_key', '/usr/sbin/postconf: warning: nonexistent_key:\
 unknown parameter')

    >>> postconf_set('smtp_sasl_password_maps', 'hash:'+passwordfile,
    ...     confdir=setup['tmpdir'], raisemode=True)[0]  # doctest: +SKIP
    0
    """
    if value is None:
        # delete the key, so the default value is used again
        if confdir:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postconf',
                    '-X', '-c', confdir, key,
                ],
                raisemode=raisemode,
            )
        else:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postconf',
                    '-X', key,
                ],
                raisemode=raisemode,
            )

        if errlvl != 0:
            msg = ('ERROR while deleting postfix configuration '
                   'key "%s"' % key)
            say.say(msg)
    else:
        # modify the key, so the new value is used
        if confdir:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postconf',
                    '-e', '-c', confdir, key + '=' + value,
                ],
                raisemode=raisemode,
            )
        else:
            errlvl, out = safe_syscall(
                [
                    '/usr/bin/sudo', '/usr/sbin/postconf',
                    '-e', key + '=' + value,
                ],
                raisemode=raisemode,
            )
        if errlvl != 0:
            msg = ('ERROR while setting postfix configuration "%s" to "%s"\n'
                   'Errorlevel: %s\nOutput: %s' % (key, value, errlvl, out))
            say.say(msg)

    value = postconf_get(key, raisemode=raisemode, confdir=confdir)[3]

    return errlvl, out, key, value


def postconf_get(key, raisemode=False, confdir=None):
    """Read out the currently configured value of a postfix configuration
    directive using the 'postconf' utility.

    >>> postconf_get('inet_interfaces', confdir=tmpdir)  # doctest: +SKIP
    (0, '', 'inet_interfaces', 'all')

    >>> postconf_get('nonexistent_key', confdir=tmpdir)  # doctest: +SKIP
    (0, '', 'nonexistent_key', '/usr/sbin/postconf: warning: nonexistent_key:\
 unknown parameter')

    >>> postconf_get('always_bcc', confdir=tmpdir)  # doctest: +SKIP
    (0, '', 'always_bcc', '')

    >>> postconf_get('foobar', confdir='nonexistent')[0]  # doctest: +SKIP
    1
    """
    msg = ''
    if confdir:
        errlvl, out = safe_syscall(
            [
                '/usr/sbin/postconf',
                '-h', '-c', confdir, key,
            ],
            raisemode=raisemode,
        )
    else:
        errlvl, out = safe_syscall(
            [
                '/usr/sbin/postconf',
                '-h', key,
            ],
            raisemode=raisemode,
        )
    value = None
    if errlvl != 0:
        msg = ('ERROR while reading out postfix configuration for '
               'key "%s"\nErrorlevel: %s\nOutput: %s' % (key, errlvl, out))
        say.say(msg)
    else:
        value = out.strip()
    return errlvl, msg, key, value


def postfix_daemon(cmd, raisemode=False, dryrun=False):
    """Send a command to the postfix mailserver process

    IMPORTANT: enable /etc/init.d/postfix in /etc/sudoers to be executable
    via 'sudo' without entering a password for the user of this function
    (e.g. zope). Only allow the parameters mentioned in the whitelist

    zope ALL=NOPASSWD:/etc/init.d/postfix reload, /etc/init.d/postifx restart

    >>> postfix_daemon(cmd='reload', raisemode=False, dryrun=True)[0]
    0

    >>> postfix_daemon(cmd='restart', raisemode=True)[0]  # doctest: +SKIP
    0

    >>> postfix_daemon(cmd='foobar', raisemode=True)  # doctest: +SKIP
    (65, 'ERROR - command "foobar" not allowed!')
    """
    whitelist = ['reload', 'restart']
    if cmd not in whitelist:
        msg = 'ERROR - command "%s" not allowed!' % cmd
        say.say(msg)
        return 65, msg

    # reload postfix
    if dryrun:
        errlvl = 0
        out = ''
    else:
        errlvl, out = safe_syscall(
            [
                '/usr/bin/sudo', '/etc/init.d/postfix',
                cmd,
            ],
            raisemode=raisemode,
        )
    if errlvl != 0:
        msg = ('ERROR occured while issuing command "%s" to postfix daemon'
               ' Errorlevel: %s\nOutput: %s' % (cmd, errlvl, out))
        say.say(msg)
    return errlvl, out


# GLOBAL settings #
# This 'whitelist' dictionary contains all allowed and tested postfix
# configuration directeives as 'keys'
# The values are either function names or lists containing a function
# name as first item, and parameters as following items
#
# Attention: all functions need to be already defined before used in the
# dictionary!
global whitelist
whitelist = {
    'mydestination': validate_domain_list,
    'relayhost': validate_relayhost,
    'smtp_sasl_auth_enable': [validate_regexp, 'yes|no'],
    'smtp_sasl_password_maps': validate_maps,
    'smtp_sasl_mechanism_filter': [
        validate_regexp,
        '^[!]{0,1}(plain|login)[ ]*(,[ ]*[!]{0,1}(plain|login))*$'
    ],
    'smtp_use_tls': [validate_regexp, 'yes|no'],
    'smtp_tls_wrappermode': [validate_regexp, 'yes|no'],
    'smtp_tls_security_level': [validate_regexp, 'may|encrypt'],
    'smtp_tls_protocols': validate_tls_protocols,
    'smtpd_use_tls': [validate_regexp, 'yes|no'],
    'smtpd_tls_wrappermode': [validate_regexp, 'yes|no'],
    'smtpd_tls_security_level': [validate_regexp, 'may|encrypt'],
    'smtpd_tls_protocols': validate_tls_protocols,
}


def doctests_setup():
    tmpdir = '/tmp/python-modules_unittest_mail'

    # clean up any leftovers
    if os.path.exists(tmpdir):
        shutil.rmtree(tmpdir)
    os.mkdir(tmpdir)

    # make copys to work on them
    # this is needed to circumvent ownership/permission problems
    # so do not test directly with the files in the repository!
    assets = os.path.abspath(os.path.join(__file__, '../assets/tests'))
    main_cf_template = os.path.join(assets, 'mail_main.cf_template')
    main_cf = tmpdir + '/main.cf'
    master_cf_template = os.path.join(assets, 'mail_master.cf_template')
    master_cf = tmpdir + '/master.cf'

    btree_mappingfile_template_db = os.path.join(
        assets, 'mail_btree_mappingfile.db')
    btree_mappingfile_db = tmpdir + '/btree_mappingfile.db'

    btree_aliasfile_template_db = os.path.join(
        assets, 'mail_btree_aliasfile.db')
    btree_aliasfile_db = tmpdir + '/btree_aliasfile.db'

    shutil.copy(main_cf_template, main_cf)
    shutil.copy(master_cf_template, master_cf)
    shutil.copy(btree_mappingfile_template_db, btree_mappingfile_db)
    shutil.copy(btree_aliasfile_template_db, btree_aliasfile_db)

    btree_mappingfile = tmpdir + '/btree_mappingfile'

    btree_aliasfile = tmpdir + '/btree_aliasfile'

    mappingfile = tmpdir + '/mappingfile'
    fd = open(mappingfile, 'w')
    fd.write('user@example.com  :foobar')
    fd.close()

    passwordfile = tmpdir + '/passwordfile'
    aliasmap = tmpdir + '/aliasmap'

    return {
        'mappingfile': mappingfile,
        'btree_mappingfile': btree_mappingfile,
        'btree_aliasfile': btree_aliasfile,
        'passwordfile': passwordfile,
        'aliasmap': aliasmap,
        'tmpdir': tmpdir,
    }


def doctests_teardown(test_setup):
    # remove files modified by doctests
    shutil.rmtree(test_setup['tmpdir'])
