#!/usr/bin/env python
# coding: utf8
#
# notifyreader.py  -  Tool to read notifies from postgresql.
#
# Copyright (C) 2019 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Fgit pulloundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import select


class NotifyReader(object):
    ''' collects notifies from a psycopg2 connection
    makes one commit on init '''

    def __init__(self, con, notifier):
        self.notifier = notifier
        self.con = con
        self.cur = con.cursor()
        self.cur.execute('listen ' + self.notifier)
        self.con.commit()
        self._notify_count = 0

    def update(self):
        ''' update notify_count '''
        self.con.poll()
        self._notify_count = 0
        i = 0
        while i < len(self.con.notifies):
            n = self.con.notifies[i]
            # check the channel. This con could be listening for more channels
            if n.channel == self.notifier:
                self._notify_count += 1
                self.con.notifies.pop(i)
                i -= 1
            i += 1

    def reset(self):
        ''' reset notify_count '''
        self._notify_count = 0

    @property
    def count(self):
        ''' number of notifies since the last update '''
        return self._notify_count

    def waitForNew(self, timeout):
        ''' wait blocking for a new notify '''
        # Perform timed or blocking read of fifo
        fd = self.con.fileno()
        readable, writable, exceptions = select.select([fd], [], [], timeout)
        # attention: there is no more filter used here like in self.notify
        self.update()

    def waitFor(self, timeout):
        '''
        Wait blocking for the first notify since last update.
        This notify may allready have been posted.
        '''
        self.update()
        if self.count == 0:
            self.waitForNew(timeout)

    def close(self):
        self.cur.close()
        self.con.close()
