#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# oidc.py  -  OpenID Connect methods
#
# Copyright (C) 2017 PerFact Innovation GmbH & Co.KG  <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

# PyJWT
import jwt
import requests
from .generic import generate_random_string, to_ustring
import six

if six.PY2:
    from urllib import urlencode
else:
    from urllib.parse import urlencode


def postReq(url, data=None, auth=None, verify=True, headers=None,
            *args, **kwargs):
    """
    Issue an HTTP POST request using the 'requests' module and return the
    answer to the caller.
    """
    res = requests.post(
        url=url,
        data=data,
        auth=auth,
        verify=verify,
        headers=headers,
        *args,
        **kwargs
    )
    return res


def getReq(url, verify=True, *args, **kwargs):
    """
    Issue an HTTP GET request using the 'requests' module and return the
    answer to the caller.
    """
    res = requests.get(url=url, verify=verify, *args, **kwargs)
    return res


def jsonReq(url, verify=True, *args, **kwargs):
    """
    Request a json from the URL and return it as a dictionary
    """
    res = getReq(
        url=url,
        verify=verify,
        *args,
        **kwargs
    )
    status_code = res.status_code
    txt = res.text
    try:
        json = res.json()
    except ValueError:
        json = None
    return {
        'text': txt,
        'json': json,
        'status_code': status_code,
        'res': res,
    }


def passwordTokenReq(url, username, password, client_id, client_secret=None,
                     scope=None, *args, **kwargs):
    """
    Use the OIDC grant type 'password' to aquire an access_token from the
    OpenID Provider (OP).

    :client_secret (optional)

    :scope (optional)
    """
    params = {
        'grant_type': 'password',
        'username': username,
        'password': password,
        'client_id': client_id,
        'client_secret': client_secret,
        'scope': scope,
    }
    res = postReq(
        url=url,
        data=params,
        *args,
        **kwargs
    )
    status_code = res.status_code
    txt = res.text
    try:
        json = res.json()
    except ValueError:
        json = None
    return {
        'text': txt,
        'json': json,
        'status_code': status_code,
        'res': res,
    }


def authReq(url, client_id, client_secret, auth_code, redirect_uri,
            verify=True, *args, **kwargs):
    '''
    Query the backend OpenID-Connect Provider (OP) to receive information about
    the login process.

    :url
    token endpoint address

    :client_id
    username

    :client_secret
    password

    :auth_code
    authorization code which was given to us on the frontchannel

    :redirect_uri
    address for redirection which was given to us on the frontchannel

    :*args
    will just be passed to the postReq function

    :**kwargs
    will just be passed to the postReq function
    '''
    # request to backend
    data = {
        'grant_type': 'authorization_code',
        'code': auth_code,
        'redirect_uri': redirect_uri,
    }
    # prepare the authentication against the server
    # the requests library takes care of converting that to HTTP Basic auth
    # using base64 encoding of username:password
    auth = (client_id, client_secret)
    res = postReq(
        url=url,
        data=data,
        auth=auth,
        verify=verify,
        *args,
        **kwargs
    )
    status_code = res.status_code
    txt = res.text
    try:
        json = res.json()
    except Exception:
        json = None
    return {
        'text': txt,
        'json': json,
        'status_code': status_code,
        'res': res,
    }


def jwkToPubKeyObj(keystring):
    '''
    The JWT library requires to pass a specific public key object
    which can be derived from a json string (JWT) like:

    {"kid":"QYVpNfd-JydP-p5JhOf3vzB9QfHzp8l0LrOLpXH1Zms",
     "kty":"RSA","alg":"RS256","use":"sig",
     "n":"g5zWXNIbDORmfPK2t6e-yQ3vIG2K7TOg4su0yiWzs0esJaiiYHLsToQWIAqGskNh\
     6JHbpo-cPQyN96lLyCSBaP9YESJ8ikZLj43g9Ue4jv4GCA2b_B90sULGkRFHd3TLPCoW1\
     dEU3BOD2V54OOo5rwwUHkaeFBDPwst27Xwye94oarDy3BQHIRDuP_nd3Pj7lwVzLqkyGP\
     FCXmTGPGfHSohW82tCbKNVqGvQTfNBrBu0we6KC6EYAszvypmbzRuhXkQitCb4-IELhcA\
     ZoZ7EQXcLSwo75O8uzfBtZbH77mJb_6YyeNmThiHxeK8xJXPJymsrB2vpa0sggkB_O2Qjiw",
     "e":"AQAB"}
    '''
    keyobj = jwt.algorithms.RSAAlgorithm.from_jwk(keystring)
    return keyobj


def parseJwt(jwtobj,
             key=None,
             verify=True,
             audience=None,
             options=None,
             algorithms=None):
    '''
    Parse/read values from a received JWT (Java Web Token).
    '''
    if key:
        if isinstance(key, str) and key.startswith('{'):
            # likely a json string which needs to be converted
            key = jwkToPubKeyObj(key)

        # try to verify the signature
        id_token = jwt.decode(
            jwt=jwtobj,
            key=key,
            verify=verify,
            options=options,
            audience=audience,
            algorithms=algorithms
        )
    else:
        id_token = jwt.decode(jwt=jwtobj, verify=verify, options=options)
    return id_token


def createJwt(id_token, key=None, alg='RS256'):
    '''
    ONLY FOR TESTING!
    Creat a JWT (Java Web Token) from a dictionary, a key and optional
    signature algorithm.

    :iss
    issuer of the token e.g. the endpoint URL

    :sub
    subject of the token (user ID or name)

    :aud
    audience that the token is intended for
    for decoding the audience must be in the list of expected/accepted
    audiences

    :nonce
    number only used once - random value which *must* not be reused

    :exp
    expiry timestamp in seconds (epoch timestamp)

    :iat
    issue timestamp in seconds (epoch timestamp)

    :algorithm
    RS256 - RSA Signature with SHA-256 (private/public keys needed)
    HS256 - HMAC with SHA-256

    :id_token
    dictionary e.g.
    {"iss": "https://server.example.com",
      "sub": "testuser",
      "aud": "some-value",
      "nonce": "xjs8Yu0-2",
      "exp": 2147382000,
      "iat": 1507212328,
    }
    '''
    if alg == 'HS256' and not key:
        raise AssertionError('A key is needed to calculate the digest')

    encoded = jwt.encode(
        payload=id_token,
        key=key,
        algorithm=alg,
    )
    return to_ustring(encoded)


def authRedir(client_id, auth_uri, redirect_uri,
              response_type='code', scope='openid',
              state=None, statelen=128, nonce=None):
    '''
    Generate a URL to send the client to the OpenID-Connect login page.
    --Input--
    :client_id
    unique identifier of this Relying Party

    :auth_uri
    URL for authentication requests at the OpenID-Provider server

    :redirect_uri
    Optional URL where the OpenID-Provider will redirect to

    :response_type
    should be 'code'

    :scope
    should be 'openid'

    :state
    optional random value, will be created if None is given
    - 'statelen' length of the state if it should be created
    - 'nonce' optional number used only once

    --Output--
    :target_url
    holds a URL including all parameters needed for an OpenID-Connect process

    :state
    represents a kind of csrf token to uniquely identify this login process
    '''
    if not state:
        state = generate_random_string(length=statelen, mode='normal')
    params = {
        'client_id': client_id,
        'state': state,
        'response_type': response_type,
        'scope': scope,
        'redirect_uri': redirect_uri,
    }
    if nonce:
        params['nonce'] = nonce

    target_url = auth_uri + '?' + urlencode(params)

    return target_url, state


def userInfoReq(url, access_token, verify=True, *args, **kwargs):
    '''
    Request additional user information from the OpenID Provider
    using the bearer token for authentication.

    :url
    Endpoint for request

    :access_token
    Bearer token from previous request in OIDC context

    :verify
    Indicates if the server certificate should be verified

    :*args
    will just be passed to the postReq function

    :**kwargs
    will just be passed to the postReq function
    '''
    headers = {'Authorization': 'Bearer ' + access_token}
    res = postReq(
        url=url,
        headers=headers,
        verify=verify,
        *args,
        **kwargs
    )
    status_code = res.status_code
    txt = res.text
    try:
        json = res.json()
    except Exception:
        json = None
    return {
        'text': txt,
        'json': json,
        'status_code': status_code,
        'res': res,
    }


def parseAuthResp(resp, aud=None, key=None, verify=True, algorithms=None):
    '''
    Parse the response of an OpenID-Connect Authentication request to extract
    useful information from the embedded JWT (Java Web Token)
    --Input--
    :resp
    Dictionary containint the server response (generated by pythons requests
    lib)

    :aud
    String containing the expected audience a.k.a the OpenID client_id of the
    RP

    :verify
    Boolean to control the verification of the JWT

    --Output--
    A dictionary containing the following keys
    :msg
    An informational message

    :status_code
    The HTTP status code of the response

    :text
    Embedded HTML of the response

    :id_token
    Dictionary which contains the results of the parsed JWT
    '''
    result = {
        'msg': None,
        'status_code': resp.get('status_code'),
        'text': resp.get('text'),
        'err': 0,
    }
    if resp.get('status_code') != requests.codes.ok:
        result['msg'] = 'The server returned an error'
        result['err'] = 6
        return result

    if not resp['json']:
        result['msg'] = 'No json in the returned response from the OpenID'
        ' Provider'
        result['err'] = 1
        return result

    # decode the json and filter out the 'id_token' and so on...
    if not isinstance(resp['json'], dict):
        result['msg'] = 'The returned json is empty'
        result['err'] = 2
        return result

    # h19t1...sJy (random value)
    result['access_token'] = resp['json'].get('access_token')

    # 7199 (seconds?)
    result['expires_in'] = resp['json'].get('expires_in')

    # e.g. 'Bearer'
    result['token_type'] = resp['json'].get('token_type')

    # e.g. 'openid'
    result['scope'] = resp['json'].get('scope')

    # V0m....Fha (random value)
    result['refresh_token'] = resp['json'].get('refresh_token')

    # JWT - java web token
    jwt = resp['json'].get('id_token')

    if not jwt:
        result['msg'] = 'No id_token a.k.a JWT was found in the JSON structure'
        result['err'] = 3
        return result

    # extract the information (claims) from the JWT (JSON web token)
    # Signature verification - how? with which key?
    # options = {'verify_aud': True}
    options = None
    if not verify:
        options = {'verify_signature': False}
    try:
        id_token = parseJwt(
            jwt,
            key=key,
            verify=verify,
            options=options,
            audience=aud,
            algorithms=algorithms
        )
    except Exception as err:
        result['msg'] = (
            'JWT could not be parsed/validated.'
            'The reason was: %s' % (str(err))
        )
        result['err'] = 5
        return result

    if not len(id_token.keys()):
        result['msg'] = (
            'The id_token a.k.a JWT did not contain any'
            ' information'
        )
        result['err'] = 4

    # The id_token can hold the following keys
    # for reference see:
    # http://openid.net/specs/openid-connect-core-1_0.html#StandardClaims
    #
    # sub                - username e.g. perfacttest1@example.com
    # aud                - audience, must be the OpenID client_id
    # realm_name         - LdapRegistry (optional)
    # iss                - issuer e.g. https://openid.example.com/oidc/
    #                      endpoint/foo
    # uniqueSecurityName - uid=perfacttest1@example.com,ou=undefined,
    #                      ou=clients,dc=example,dc=com
    # at_hash            - i6PU...g1A (random value?)
    # groupIds           - [u'cn=serviceconsumer,ou=roles,dc=example,dc=com']
    # exp                - expiry timestamp 1509384970 (unix epoch timestamp)
    # iat                - issued at timestamp 1509377770
    #                      (unix epoch timestamp)
    result['id_token'] = id_token

    result['err'] = 0
    return result
