#
# sound.py  -  Methods for creating and converting sound files.
#
# Copyright (C) 2016 Jan Jockusch <jan.jockusch@perfact-innovation.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#

#
# Required software
#
# espeak (for speech conversion)
# ffmpeg (for sound format conversion)
#

import tempfile

from .generic import safe_syscall


def call_espeak(sentence, speed=160, voice='de', format='wav'):
    '''Create spoken output for the given sentence.

    You can formulate the sentence in the language needed:
    >>> sentence = "Bitte abladen in Regal 12 A... Ebene 15"

    And then let espeak convert it into spoken language.
    >>> (call_espeak(sentence)[0:20] ==
    ...  b'RIFF\\xfe\\x96\\x02\\x00WAVEfmt \\x10\\x00\\x00\\x00'
    ... )  # doctest: +SKIP
    True

    >>> (call_espeak(sentence, format='mp3')[0:18] ==
    ...  b'ID3\\x04\\x00\\x00\\x00\\x00\\x00#TSSE\\x00\\x00\\x00\\x0f'
    ... )  # doctest: +SKIP
    True

    >>> (call_espeak(sentence, format='ogg')[0:10] ==
    ...  b'OggS\\x00\\x02\\x00\\x00\\x00\\x00')  # doctest: +SKIP
    True

    '''
    legal_formats = ['wav', 'mp3', 'ogg']
    assert format in legal_formats, \
        "Illegal format. Use one of {}".format(legal_formats)

    tempdir = tempfile.mkdtemp()
    wavfile = tempdir + '/out.wav'
    retcode, output = safe_syscall(
        ['espeak', '-s', str(speed), '-v', voice,
         '-w', wavfile, sentence], raisemode=True)
    if format == 'wav':
        data = open(wavfile, 'rb').read()
        return data

    outfile = tempdir + '/out.' + format
    retcode, output = safe_syscall(
        ['ffmpeg', '-i', wavfile, outfile],
        raisemode=True)

    data = open(outfile, 'rb').read()
    return data


if __name__ == '__main__':
    import doctest
    doctest.testmod()
