import unittest
import time
import tempfile
import threading
import os
import perfact.cert
from perfact.generic import safe_syscall
from ..file import tmp_cleanup


# Following helper functions used by test methods
def create_and_fill_tmpd(file_names):
    tmpdir = tempfile.mkdtemp()
    for file_name in file_names:
        file_path = os.path.join(tmpdir, file_name)
        open(file_path, 'w').close()

    return tmpdir


def acquire_flock(ca_path, timeout):
    lockfile = os.path.join(ca_path, perfact.cert.openssl_lockfile)

    lock_cmds = ['flock', '--wait', '5', lockfile]
    lock_cmds.extend(['sleep', str(timeout)])

    safe_syscall(lock_cmds)


def fake_os_path_isfile(file):
    '''
    Fake os.path.isfile to always trigger the "repair" action
    for the function ca_directory_repair
    '''
    if file.endswith('.new'):
        return True
    return False


class TestCert(unittest.TestCase):
    def test_ca_directory_integrity(self):
        test_cases = {
            ('index.txt', 'index.txt.attr', 'serial'): True,
            ('index.txt.new', 'index.txt.attr', 'serial'): False,
            ('index.txt', 'index.txt.attr'): False,
        }
        for files, result in test_cases.items():
            tmpdir = create_and_fill_tmpd(files)

            integrity = perfact.cert.ca_directory_check_integrity(
                ca_path=tmpdir
            )
            self.assertIs(
                result,
                integrity,
                'Error while checking the integrity of the CA directory'
            )

            tmp_cleanup(tmpdir)

    def test_ca_directory_repair(self):
        test_cases = {
            ('index.txt', 'index.txt.attr', 'serial'): False,
            ('index.txt.new', 'index.txt.attr', 'serial'): True,
            ('index.txt', 'index.txt.attr'): False,
        }
        for files, result in test_cases.items():
            tmpdir = create_and_fill_tmpd(files)

            # Check if a repair action was performed
            repaired = perfact.cert.ca_directory_repair(ca_path=tmpdir)
            self.assertIs(
                result,
                repaired,
                'Error during the repair of the CA directory'
            )

            integrity = perfact.cert.ca_directory_check_integrity(
                ca_path=tmpdir
            )

            # Integrity should be restored if a repair was performed
            if repaired:
                self.assertTrue(
                    integrity,
                    'Error while checking the integrity of the CA directory '
                    'after a repair was performed'
                )

            tmp_cleanup(tmpdir)

    def test_ca_directory_repair_mv_fail(self):
        tmpdir = create_and_fill_tmpd(
            ['index.txt', 'serial', 'index.txt.attr']
        )

        # "Mock" os.path.isfile to force a failing mv on a file
        orig_isfile = perfact.cert.os.path.isfile
        perfact.cert.os.path.isfile = fake_os_path_isfile

        # Because we ignore a 'mv' fail the return value
        # should still be true
        repaired = perfact.cert.ca_directory_repair(ca_path=tmpdir)
        self.assertTrue(
            repaired,
            'Error: "mv" failure was not ignored'
        )

        perfact.cert.os.path.isfile = orig_isfile
        tmp_cleanup(tmpdir)

    def test_ca_directory_repair_flock_timeout(self):
        # Test raise if a lock could not be acquired using flock
        tmpdir = create_and_fill_tmpd(
            ('index.txt.new', 'index.txt.attr', 'serial')
        )

        locking = threading.Thread(
            target=acquire_flock,
            args=(tmpdir, 4)
        )
        locking.start()
        # Wait for the thread to grab the lock
        time.sleep(1)

        # The CA directory repair must fail because a lock
        # can not be acquired due to the thread locking
        with self.assertRaises(AssertionError):
            perfact.cert.ca_directory_repair(
                ca_path=tmpdir,
                flock_timeout=2
            )

        locking.join()

        tmp_cleanup(tmpdir)
