#!/usr/bin/env python
# -*- coding: utf-8 -*-

import unittest
import stat
import six
from .. import file

if six.PY2:
    import mock
else:
    from unittest import mock


class TestFile(unittest.TestCase):

    @mock.patch("os.path.exists")
    def test_upload_asset_filehandle(self, mock_path_exists):
        # Test values
        test_string1 = "Test123"
        test_string2 = "Another test"
        test_filename = "cool_file.txt"
        test_subpath = "somefolder/somefolder2"
        full_path = ("/vol/zope_files/assets/"
                     "somefolder/somefolder2/cool_file.txt")

        # Test mocks
        test_data = mock.MagicMock()
        # Values for multiple calls
        # Empty string acts as EOF
        test_data.read.side_effect = [test_string1, test_string2, ""]
        mock_path_exists.return_value = True

        with mock.patch.object(file, 'open') as m_open:
            # Prepare mocked 'open()'
            outfilegen = mock.MagicMock()
            outfile = mock.MagicMock()
            outfilegen.__enter__.return_value = outfile
            m_open.return_value = outfilegen

            # Function call, we expect no error
            ret_val = file.upload_asset(test_data,
                                        test_filename,
                                        test_subpath)

            # Validating calls
            m_open.assert_called_once_with(full_path, "wb")
            outfile.write.assert_any_call(test_string1)
            outfile.write.assert_called_with(test_string2)
            outfilegen.__exit__.assert_called_once()

            cc = outfile.write.call_count
            assert cc == 2, "Expected 2 calls, got " + str(cc)

        # Validate return value
        self.assertEqual(ret_val, full_path)

    @mock.patch("os.path.exists")
    def test_upload_asset_bytes(self, mock_path_exists):
        # Test values
        test_filename = "cool_file.txt"
        test_subpath = "somefolder/somefolder2"
        full_path = ("/vol/zope_files/assets/"
                     "somefolder/somefolder2/cool_file.txt")

        # Test mocks, test_data mimics a bytes-object
        test_data = mock.MagicMock(spec=bytes)
        mock_path_exists.return_value = True

        with mock.patch.object(file, 'open') as m_open:
            # Prepare mocked 'open()'
            outfilegen = mock.MagicMock()
            outfile = mock.MagicMock()
            outfilegen.__enter__.return_value = outfile
            m_open.return_value = outfilegen

            # Function call, we expect no error
            ret_val = file.upload_asset(test_data,
                                        test_filename,
                                        test_subpath)

            # Validating calls
            m_open.assert_called_once_with(full_path, "wb")
            outfile.write.assert_called_once_with(test_data)
            outfilegen.__exit__.assert_called_once()

        # Validating returned result
        self.assertEqual(ret_val, full_path)

    @mock.patch("os.path.exists")
    def test_upload_asset_incorrect_paths(self, mock_path_exists):
        # Test nonexistant paths
        test_filename = "cool_file.txt"
        test_subpath = "somefolder/somefolder2"

        # Test mocks, os.path.exists has to return False
        test_data = mock.MagicMock()
        test_data.read.return_value = ""
        mock_path_exists.return_value = False

        # Function call, we expect an error because our path doesnt exist
        with self.assertRaises(AssertionError):
            file.upload_asset(test_data, test_filename, test_subpath)

        # Test forbidden paths, we must not leave /vol/zope_files/assets
        mock_path_exists.return_value = True
        test_fbdn_subpaths = ["/etc/", "../foo"]

        with mock.patch.object(file, 'open') as m_open:
            for fbdn_path in test_fbdn_subpaths:
                with self.assertRaises(AssertionError):
                    # Function call, we expect an error.
                    # We must not open a file for such subpaths
                    file.upload_asset(test_data, test_filename, fbdn_path)
                    m_open.assert_not_called()

    @mock.patch("os.path.exists")
    def test_upload_asset_invalid_filenames(self, mock_path_exists):
        # Test values, filename with a "/" and path shenanigans
        # We must not leave /vol/zope_files/assets
        test_filenames = ["cool/file\\.txt", "../testfile", "/etc/my_file"]
        test_subpath = ""

        # Test mocks
        test_data = mock.MagicMock()
        test_data.read.return_value = ""
        mock_path_exists.return_value = True

        with mock.patch.object(file, 'open') as m_open:
            for inv_fn in test_filenames:
                # Function call, we expect an error,
                # because our filename is invalid
                with self.assertRaises(AssertionError):
                    file.upload_asset(test_data, inv_fn, test_subpath)
            m_open.assert_not_called()

    def test_asset_passthrough(self):
        with mock.patch.object(file, 'os') as m_os, \
                mock.patch.object(file, 'get_mimetype') as m_get_mimetype, \
                mock.patch.object(file, 'open') as m_open:
            stat_result = mock.Mock()
            stat_result.configure_mock(st_mode=stat.S_IFREG, st_size=6)
            m_os.path.join.return_value = (
                '/vol/zope_files/assets/tests/testfile.txt'
            )
            m_os.lstat.return_value = stat_result
            m_os.access.return_value = True
            m_get_mimetype.return_value = 'text/plain'
            infilegen = mock.MagicMock()
            infile = mock.Mock()
            infilegen.__enter__.return_value = infile
            infile.read.side_effect = [b'ABCDE\n', b'']
            m_open.return_value = infilegen
            RESPONSE = mock.Mock()
            output = file.asset_passthrough(
                '/vol/zope_files/assets',
                ['tests', 'testfile.txt'],
                RESPONSE
            )
            self.assertEqual(output, None)
            m_os.path.join.assert_called_with(
                '/vol/zope_files/assets', 'tests', 'testfile.txt')
            m_os.lstat.assert_called_with(
                '/vol/zope_files/assets/tests/testfile.txt')
            m_os.access.assert_called_with(
                '/vol/zope_files/assets/tests/testfile.txt', m_os.R_OK)
            m_get_mimetype.assert_called_with(
                '/vol/zope_files/assets/tests/testfile.txt')
            RESPONSE.setHeader.assert_has_calls([
                mock.call('Content-Length', '6'),
                mock.call('Content-Type', 'text/plain'),
                mock.call('Cache-Control', 'max-age=36000')
            ])
            m_open.assert_called_with(
                '/vol/zope_files/assets/tests/testfile.txt', 'rb')
            infile.read.assert_has_calls([
                mock.call(1024*1024),
                mock.call(1024*1024),
            ])
            RESPONSE.write.assert_has_calls([
                mock.call(b'ABCDE\n'),
            ])
