#!/usr/bin/env python
# -*- coding: utf-8 -*-

import unittest

from .. import graphics
from .mock_helper import ModulePatches, mock


class TestGraphics(unittest.TestCase):

    def test_call_ploticus(self):
        test_input = u'// Testing ploticus ÄÖÜ'

        # Test for assertion of legal format
        self.assertRaises(
            ValueError, graphics.call_ploticus,
            '', 'xng',
        )

        for input_variant in (test_input, test_input.encode('latin1')):
            # Patch all I/O calls within "call_ploticus"
            with ModulePatches(graphics, [
                        'safe_syscall', 'tempfile', 'open',
                        'eps_to_png', 'convert_tmpfile',
                    ]) as mobj:

                # Define return values for mocks
                mobj.tempfile.mkdtemp.return_value = '/tmp/test'
                mobj.safe_syscall.return_value = (
                    0, b'%!PS-Adobe-3.0 EPSF-3.0\n...'
                )
                mobj.convert_tmpfile.return_value = (
                    b'%!PS-Adobe-3.0 EPSF-3.0\n...'
                )

                # Do the call proper
                output = graphics.call_ploticus(
                    input_variant, format='eps', dpi=100
                )

                # Check how the mocks were called
                mobj.safe_syscall.assert_called_once_with([
                    '/usr/bin/ploticus', '/tmp/test/in.txt',
                    '-eps', '-o', '/tmp/test/out.eps',
                ])
                mobj.open.assert_has_calls([
                    mock.call('/tmp/test/in.txt', 'wb'),
                    mock.call().write(b'// Testing ploticus \xc4\xd6\xdc'),
                    mock.call().flush(),
                    mock.call().close(),
                ])
                mobj.convert_tmpfile.assert_called_once_with(
                    'eps', '/tmp/test'
                )
                mobj.eps_to_png.assert_not_called()
                self.assertTrue(
                    output.startswith(b'%!PS-Adobe-3.0 EPSF-3.0\n')
                )

        with ModulePatches(graphics, [
                    'safe_syscall', 'tempfile', 'open',
                    'eps_to_png', 'convert_tmpfile',
                ]) as mobj:

            mobj.tempfile.mkdtemp.return_value = '/tmp/test'
            mobj.safe_syscall.return_value = (
                0, b'%!PS-Adobe-3.0 EPSF-3.0\n'
            )
            mobj.convert_tmpfile.return_value = b'PNG<mock>'
            mobj.eps_to_png.return_value = b'PNG<mock>'
            mobj.open().read.return_value = b'OUT<mock>'

            output = graphics.call_ploticus(test_input, format=None, dpi=100)

            mobj.safe_syscall.assert_called_once_with([
                '/usr/bin/ploticus', '/tmp/test/in.txt',
                '-eps', '-o', '/tmp/test/out.eps',
            ])
            mobj.open.assert_has_calls([
                mock.call(),
                mock.call('/tmp/test/in.txt', 'wb'),
                mock.call().write(b'// Testing ploticus \xc4\xd6\xdc'),
                mock.call().flush(),
                mock.call().close(),
                mock.call('/tmp/test/out.eps', 'rb'),
                mock.call().read(),
                mock.call().close(),
                mock.call('/tmp/test/out.png', 'wb'),
                mock.call().write(b'PNG<mock>'),
                mock.call().close(),
            ])
            mobj.convert_tmpfile.assert_called_once_with('png', '/tmp/test')
            mobj.eps_to_png.assert_called_once_with(
                b'OUT<mock>', res=100
            )
            self.assertEqual(output, b'PNG<mock>')

    def test_call_gnuplot(self):
        with ModulePatches(graphics, [
                    'tempfile', 'open', 'subprocess'
                ]) as mobj:
            mobj.tempfile.mkdtemp.return_value = '/tmp/test'
            mobj.subprocess.PIPE = 'PIPE'
            mobj.open().read.return_value = b'graph'

            output = graphics.call_gnuplot('control', 'data')
            mobj.open.assert_has_calls([
                mock.call('/tmp/test/data', 'w'),
                mock.call().write('data'),
                mock.call().close(),
                mock.call('/tmp/test/output', 'r'),
                mock.call().read(),
                mock.call().close(),
            ])
            mobj.subprocess.assert_has_calls([
                mock.call.Popen('/usr/bin/gnuplot', cwd='/tmp/test',
                                stdin='PIPE'),
                mock.call.Popen().communicate('control'),
            ])
            mobj.tempfile.mkdtemp.assert_called_once_with()
            self.assertEqual(output, b"graph")

    def test_call_inkscape(self):
        with ModulePatches(graphics, [
                'safe_syscall', 'convert_tmpfile', 'tempfile', 'open'
        ]) as mobj:

            tmpdir = '/tmp/test'
            mobj.tempfile.mkdtemp.return_value = tmpdir
            mobj.safe_syscall.return_value = (
                0, b'PNG'
            )

            graphics.call_inkscape(data='data', format='png')
            mobj.safe_syscall.assert_called_with(
                [
                    '/usr/bin/inkscape',
                    '-o',
                    '%s/out.png' % tmpdir,
                    '%s/in.svg' % tmpdir,
                ],
                raisemode=mock.ANY
            )
