#!/usr/bin/python
# -*- coding: utf-8 -*-

import os
import pytest

from email import message_from_string

from ..mail import mail_receive, postconf_get, postconf_main
from ..mail import postmap_create, postmap_update
from ..mail import postalias_create, postalias_update
from ..mail import mail_decodeheader
from .. import fileassets
from .. import dbconn
from ..generic import safe_syscall


@pytest.mark.skip(reason="Requires a special user and sudo privileges")
def test_all():
    """Automatically test all features of this module using either unattended
    or interactive mode. Call the mail.py module with the '--interactive'
    switch
    """
    # WARNING - this should be run as user 'zope'!
    uid = os.getuid()
    assert uid == 1005, (
        "You need to run as user 'zope' (uid=1005)! \n"
        "Your uid is %s. As root try: 'sudo -u zope python mail.py'" % uid
    )


@pytest.mark.skip(reason="Requires a special user and sudo privileges")
def test_mail_receive():
    """ Test mail receive using SMTP delivery on localhost to
    special user.
    This creates database entries and deletes them again, based on the
    choosen mode (interactive|unattended).
    Queries for manual deletion will be printed to STDOUT
    """
    import smtplib
    from email.mime.application import MIMEApplication
    from email.mime.multipart import MIMEMultipart
    from email.mime.text import MIMEText

    from .generic import generate_random_string

    email_paths = [
        './assets/tests/mail_WithAttachment.eml',
        './assets/tests/mail_WithoutAttachment.eml',
        './assets/tests/mail_WithSpecialsInHeader.eml',
    ]
    print("please delete the generated mail entries in appemail using the"
          " following queries:")
    queries = []
    for email_path in email_paths:
        fd = open(email_path, 'rb')
        appemail_id = mail_receive(fd)
        query = "delete from appemail where appemail_id = %d;" % appemail_id
        queries.append(query)
        print(query)
        fd.close()

    # again with optional parameter for bcc headers
    for email_path in email_paths:
        fd = open(email_path, 'rb')
        appemail_id = mail_receive(
            fd, additional_bcc_headers=['Bcc', 'Delivered-To']
        )
        query = "delete from appemail where appemail_id = %d;" % appemail_id
        queries.append(query)
        print(query)
        fd.close()

    # Integration test using the local SMTP-server

    # make sure the user part of email_address exists in /etc/aliases
    # or is a system user
    mail_from = 'unittest@perfact.de'
    rcpt_to = 'zope@localhost'
    randomized = generate_random_string()

    msg = MIMEMultipart()
    msg['From'] = mail_from
    msg['To'] = rcpt_to
    # msg['Date'] = formatdate(localtime=True)
    subject = 'Unit-Test without attachment: ' + randomized
    msg['Subject'] = subject

    msg.attach(MIMEText('This is a unit test from mail.py'))

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(
        mail_from,
        [rcpt_to, ],
        msg.as_string()
    )

    query = (
        "delete from appemail where appemail_id > %d"
        " and appemail_sender = \'%s\'"
        " and appemail_receiver = \'%s\'"
        " and appemail_subject = \'%s\';" % (
            appemail_id,
            mail_from,
            rcpt_to,
            subject,
        )
    )
    queries.append(query)
    print(query)

    # with attachment
    image_filename = 'manual-image.png'
    image = fileassets['tests.manual-image']
    msg = MIMEMultipart()
    msg['From'] = mail_from
    msg['To'] = rcpt_to
    # msg['Date'] = formatdate(localtime=True)
    subject = 'Unit-Test with attachment: ' + randomized
    msg['Subject'] = subject

    msg.attach(MIMEText('This is another unit test from mail.py'))

    attachment = MIMEApplication(image, Name=image_filename)
    attachment['Content-Disposition'] = (
        'attachment; filename="%s"' % image_filename
    )

    msg.attach(attachment)

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(
        mail_from,
        [rcpt_to, ],
        msg.as_string()
    )

    query = (
        "delete from file where file_id in \n"
        " (select file_id \n"
        "    from file \n"
        "    join appemailxfile \n"
        "      on appemailxfile_file_id = file_id \n"
        "    join appemail \n"
        "      on appemail_id = appemailxfile_appemail_id \n"
        "   where appemail_id > %d \n"
        "     and appemail_sender = \'%s\' \n"
        "     and appemail_receiver = \'%s\' \n"
        "     and appemail_subject = \'%s\');\n" % (
            appemail_id,
            mail_from,
            rcpt_to,
            subject,
        )
    )
    queries.append(query)
    print(query)

    query = (
        "delete from appemailxfile where appemailxfile_appemail_id in \n"
        " ( select appemail_id \n"
        "    from appemail \n"
        "   where appemail_id > %d \n"
        "     and appemail_sender = \'%s\' \n"
        "     and appemail_receiver = \'%s\' \n"
        "     and appemail_subject = \'%s\'); \n" % (
            appemail_id,
            mail_from,
            rcpt_to,
            subject,
        )
    )
    queries.append(query)
    print(query)

    query = (
        "delete from appemail where appemail_id > %d \n"
        " and appemail_sender = \'%s\' \n"
        " and appemail_receiver = \'%s\' \n"
        " and appemail_subject = \'%s\'; \n" % (
            appemail_id,
            mail_from,
            rcpt_to,
            subject,
        )
    )
    queries.append(query)
    print(query)

    autodelete = 'y'

    if autodelete.lower() == 'y':
        for query in queries:
            dbconn.execute(query)
        dbconn.commit()
        print("Done - deleted DB entries again")
    else:
        print("Done - did not delete anything")


@pytest.mark.skip(reason="Requires a special user and sudo privileges")
def test_mail_configure():
    """Check manipulation of postfix configuration files and settings
    """
    import os
    import shutil
    import tempfile

    print('')
    print('Starting postfix configuration manipulation test sequence')

    # make a temporary directory first to store our files!
    tempdir = tempfile.mkdtemp()
    print('Using tempdir: %s' % tempdir)

    main_cf_template = './assets/tests/mail_main.cf_template'
    master_cf_template = './assets/tests/mail_master.cf_template'

    shutil.copy(main_cf_template, tempdir + '/main.cf')
    shutil.copy(master_cf_template, tempdir + '/master.cf')

    print('Testing configuration setting to enable TLS for sending')
    postconf_main('smtp_use_tls', 'yes', confdir=tempdir, raisemode=True)
    res = postconf_get('smtp_use_tls', confdir=tempdir, raisemode=True)
    value = res[3]
    assert value == 'yes', \
        'Error while setting smtp_use_tls to yes. Got %s' % str(res)

    print('Testing configuration settint to use SASL authentication '
          'for sending')
    key = 'smtp_sasl_auth_enable'
    new_value = 'yes'
    postconf_main(key, new_value, confdir=tempdir, raisemode=True)
    res = postconf_get(key, confdir=tempdir, raisemode=True)
    value = res[3]
    assert value == new_value, \
        'Error while setting %s to %s. Got %s' % (key, new_value, str(res))

    print('Testing configuration setting to use specific SASL authentication'
          ' mechnisms for for sending mail')
    key = 'smtp_sasl_mechanism_filter'
    new_value = 'plain, login'
    postconf_main(
        key=key, value=new_value,
        confdir=tempdir, raisemode=True
    )
    res = postconf_get(key=key, confdir=tempdir, raisemode=True)
    value = res[3]
    assert value == new_value, (
        'Error while setting %s to %s. Got %s' % (key, new_value, str(res))
    )

    print('Testing generation of mapping file for SASL authentication')
    new_file = tempdir+'/relay_passwd'
    file_type = 'hash'
    postmap_create(path=new_file, maptype=file_type,
                   raisemode=True, confdir=tempdir)
    assert os.path.exists(new_file), (
        'Mapping file %s does not exist!' % new_file
    )

    print('Testing update of mapping file with new password')
    key = 'server.example.com'
    value = 'myuser:mypassword'
    postmap_update(path=new_file, key=key, value=value,
                   raisemode=True, confdir=tempdir)
    fd = open(new_file, 'r')
    line = fd.read()
    fd.close()
    errlvl, out = safe_syscall(['/usr/sbin/postmap', '-q', key,
                                file_type + ':' + new_file],
                               raisemode=True)
    out = out.strip()
    assert out == value, (
        'Unexpected value in mapping file %s. Expected: %s.'
        'Errorlevel: %s, Output: %s' % (new_file, value, errlvl, out)
    )

    print('Testing configuration setting to use a mapping file to read'
          ' the SASL authentication information from')
    key = 'smtp_sasl_password_maps'

    new_value = 'hash:'+new_file
    postconf_main(
        key=key, value=new_value,
        confdir=tempdir, raisemode=True
    )
    res = postconf_get(key=key, confdir=tempdir, raisemode=True)
    value = res[3]
    assert value == new_value, \
        'Error while setting %s to %s. Got %s' % (key, new_value, str(res))

    print('Testing configuration setting to enforce TLS usage for sending')
    key = 'smtp_tls_security_level'
    new_value = 'encrypt'
    postconf_main(
        key=key, value=new_value,
        confdir=tempdir, raisemode=True
    )
    res = postconf_get(key=key, confdir=tempdir, raisemode=True)
    value = res[3]
    assert value == new_value, (
        'Error while setting %s to %s. Got %s' % (key, new_value, str(res))
    )

    new_file = tempdir+'/my_aliases'
    file_type = 'hash'
    # write mapping file for alias maps
    postalias_create(path=new_file, maptype=file_type,
                     raisemode=True, confdir=None)
    assert os.path.exists(new_file), (
        'Alias file %s does not exist!' % new_file
    )

    print('Testing update of alias entry in alias file')
    key = 'foobar'
    value = 'batz'
    postalias_update(path=new_file, key=key, value=value,
                     raisemode=False, confdir=None)

    errlvl, out = safe_syscall(['/usr/sbin/postalias', '-q', key, new_file],
                               raisemode=True)
    out = out.strip()
    assert out == value, (
        'Unexpected value in alias file %s. Expected: %s.'
        'Errorlevel: %s, Output: %s' % (new_file, value, errlvl, out)
    )

    print('Testing conversion of already filled mapping file')
    new_file = tempdir+'/prefilled_map'
    file_type = 'btree'
    key = 'another.server.example.com'
    value = 'superuser:secretpasswd'
    fd = open(new_file, 'w')
    line = key + '    ' + value
    fd.write(line)
    fd.close()
    postmap_create(path=new_file, maptype=file_type,
                   raisemode=True, confdir=tempdir)
    errlvl, out = safe_syscall(['/usr/sbin/postmap', '-q', key,
                                file_type + ':' + new_file], raisemode=True)
    out = out.strip()
    assert out == value, (
        'Unexpected value in alias file %s. Expected: %s.'
        'Errorlevel: %s, Output: %s' % (new_file, value, errlvl, out)
    )

    # test conversion of already filled alias file
    new_file = tempdir+'/prefilled_alias'
    file_type = 'btree'
    key = 'unittest'
    value = 'zope'
    fd = open(new_file, 'w')
    line = key + ':     ' + value
    fd.write(line)
    fd.close()
    postmap_create(path=new_file, maptype=file_type,
                   raisemode=True, confdir=tempdir)
    errlvl, out = safe_syscall(['/usr/sbin/postalias', '-q', key + ':',
                                file_type + ':' + new_file], raisemode=True)
    out = out.strip()
    assert out == value, (
        'Unexpected value in alias file %s. Expected: %s.'
        'Errorlevel: %s, Output: %s' % (new_file, value, errlvl, out)
    )
    # remove files modified by doctests
    shutil.rmtree(tempdir)


def test_decodeheader():
    """
    Check that decoding headers works also with non-ASCII characters and
    Python3.
    """
    headers = [
        "=?UTF-8?Q?Betreff_mit_=c3=9f?=",
        'test@perfact.de',
        '=?utf-8?b?w58=?= =?utf-8?b?w58=?=',
    ]
    decoded_headers = [mail_decodeheader(header) for header in headers]
    expected = [
        'Betreff mit ß',
        'test@perfact.de',
        'ßß',
    ]
    assert expected == decoded_headers


def test_decodefilename():
    """
    Create a mail with an attachment that has a non-ASCII filename and make
    sure it is correctly interpreted.
    """

    encoded_fnames = [
        "UTF-8''%74%65%73%74%C3%BC%2E%70%6E%67",
        '"=?iso-8859-1?Q?test=FC.png?="',
    ]
    for fname in encoded_fnames:
        msg = "Content-Disposition: attachment; filename*=" + fname
        msg = message_from_string(msg.replace('\n', '\r\n'))
        fname = mail_decodeheader(list(msg.walk())[0].get_filename())
        assert fname == "testü.png"
