#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# test_network.py  -  Tests perfact.network
#
# Copyright (C) 2019 PerFact Innovation GmbH & Co. KG <info@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

import unittest
from ..network import InterfacesParser
from ..fileassets import fileassets
import tempfile
import os
import doctest
from .. import network


def load_tests(loader, tests, ignore):
    tests.addTests(doctest.DocTestSuite(network, optionflags=doctest.ELLIPSIS))
    return tests


class TestInterfacesParser(unittest.TestCase):

    # this will activate printing of unlimited lines when results
    # of unit tests differ
    maxDiff = None

    base_content = fileassets['tests.network_ip-base_content']
    BASE_CONTENT_PARSE_RESULT = fileassets['tests.network_ip-base_result']

    allow_auto_content = fileassets['tests.network_ip-allow_auto_content']
    ALLOW_AUTO_PARSE_RESULT = fileassets['tests.network_ip-allow_auto_result']

    allow_hotplug_content = fileassets[
        'tests.network_ip-allow_hotplug_content']
    ALLOW_HOTPLUG_PARSE_RESULT = fileassets[
        'tests.network_ip-allow_hotplug_result'
    ]

    source_content = fileassets['tests.network_ip-source_content']
    SOURCE_CONTENT_PARSE_RESULT = fileassets['tests.network_ip-source_result']

    virtual_ifs_content = fileassets['tests.network_ip-virtual_ifs_content']
    VIRTUAL_IFS_PARSE_RESULT = fileassets[
        'tests.network_ip-virtual_ifs_result'
    ]

    def test_base(self):
        ''' Parse / interpret a basic/simple configuration
        '''
        np = InterfacesParser(content=self.base_content)
        r = np.get()
        self.assertEqual(len(r), 10)

        interfaces = np.get_interfaces()
        self.assertEqual(len(interfaces), 5)
        self.assertEqual(interfaces.get("eth0").get("method"), "manual")
        self.assertEqual(interfaces.get("eth1").get("method"), "dhcp")
        self.assertEqual(interfaces.get("eth2").get("method"), "static")
        self.assertEqual(interfaces.get("eth2").get("options").get(
            "address"), "172.16.200.1")

        output = np.format()

        self.assertEqual(output, self.BASE_CONTENT_PARSE_RESULT)

    def test_read_config(self):
        ''' Read file from disk
        '''
        tmpdir = tempfile.mkdtemp()
        inpath = os.path.join(tmpdir, 'interfaces_test_in')
        with open(inpath, 'w') as infile:
            infile.write(self.base_content)
        ip = InterfacesParser(infile=inpath)
        output = ip.format()
        self.assertEqual(output, self.BASE_CONTENT_PARSE_RESULT)

    def test_parser(self):
        '''Basic parser test added to make debugging easier.

        If it fails, it prints the structure phase of the parser and the result
        of get_interfaces(), which helps greatly in understanding what goes
        wrong.
        '''
        config = '''\
auto eth0
    iface eth0 inet static
    address 192.168.1.1

'''
        parser = InterfacesParser(content=config)
        structure = parser.get()
        # These print statements actually help when building and debugging the
        # parser, so they are left here.
        print(structure)
        interfaces = parser.get_interfaces()
        print(interfaces)
        self.assertEqual(
            interfaces.get('eth0').get('method'),
            'static'
        )
        self.assertEqual(
            interfaces.get('eth0').get('options').get('address'),
            '192.168.1.1'
        )

    def test_allow_auto(self):
        np = InterfacesParser(content=self.allow_auto_content)
        r = np.get()
        # Length is exptected to be amount of interfaces * 2
        self.assertEqual(len(r), 8)

        interfaces = np.get_interfaces()
        self.assertEqual(len(interfaces), 4)

        self.assertEqual(interfaces.get("eth0").get("method"), "dhcp")
        self.assertEqual(interfaces.get("eth2").get("method"), "dhcp")
        self.assertEqual(interfaces.get("eth1").get("method"), "static")
        self.assertEqual(interfaces.get("lo").get("method"), "loopback")

        address = interfaces.get("eth1").get("options").get("address")
        self.assertEqual(address, "172.16.1.1")

        self.assertEqual(interfaces.get("eth0").get("allow-auto"), True)
        self.assertEqual(interfaces.get("eth1").get("allow-auto"), True)
        self.assertEqual(interfaces.get("lo").get("allow-auto"), None)
        self.assertEqual(interfaces.get("lo").get("auto"), True)

        output = np.format()
        self.assertEqual(output, self.ALLOW_AUTO_PARSE_RESULT)

    def test_allow_hotplug(self):
        np = InterfacesParser(content=self.allow_hotplug_content)
        r = np.get()
        self.assertEqual(len(r), 6)

        interfaces = np.get_interfaces()
        self.assertEqual(len(interfaces), 3)

        self.assertEqual(interfaces.get("eth0").get("method"), "dhcp")
        self.assertEqual(interfaces.get("eth1").get("method"), "static")
        self.assertEqual(interfaces.get("lo").get("method"), "loopback")

        output = np.format()
        self.assertTrue(all(not line.startswith("allow-hotplug")
                            for line in output.splitlines()))
        self.assertEqual(output, self.ALLOW_HOTPLUG_PARSE_RESULT)

    def test_source(self):
        np = InterfacesParser(content=self.source_content)
        r = np.get()
        self.assertEqual(len(r), 6)

        interfaces = np.get_interfaces()
        self.assertEqual(len(interfaces), 3)

        self.assertEqual(interfaces.get("eth0").get("method"), "dhcp")
        self.assertEqual(interfaces.get("eth1").get("method"), "static")
        self.assertEqual(interfaces.get("lo").get("method"), "loopback")

        output = np.format()
        self.assertTrue(all(not line.startswith("source")
                            for line in output.splitlines()))
        self.assertEqual(output, self.SOURCE_CONTENT_PARSE_RESULT)

    def test_virtual_interfaces(self):
        np = InterfacesParser(content=self.virtual_ifs_content)
        r = np.get()
        self.assertEqual(len(r), 10)

        interfaces = np.get_interfaces()
        self.assertEqual(len(interfaces), 5)

        self.assertEqual(interfaces.get("eth0").get("method"), "static")
        self.assertEqual(interfaces.get("eth1").get("method"), "dhcp")
        self.assertEqual(interfaces.get("eth0:1").get("method"), "static")
        self.assertEqual(interfaces.get("eth0:2").get("method"), "static")

        eth01_opts = interfaces.get("eth0:1").get("options")
        eth02_opts = interfaces.get("eth0:2").get("options")
        self.assertEqual(eth01_opts.get("address"), "192.168.42.84")
        self.assertEqual(eth01_opts.get("netmask"), "255.255.252.0")
        self.assertEqual(eth02_opts.get("address"), "192.168.42.85")
        self.assertEqual(eth02_opts.get("netmask"), "255.255.252.0")

        output = np.format()
        self.assertEqual(output, self.VIRTUAL_IFS_PARSE_RESULT)
