#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# zcode_analysis  -  Helpers to analyze zope application code
#
# Copyright (C) 2022 Lars Bergmann <lars.bergmann@perfact.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
'''
zcode_analysis

A python module to analyze application code living in dbutils-zoperepo.
'''

import os
import re
import argparse
from perfact.generic import safe_syscall


# DEFAULTS - implement as globals to allow monkey patching
REPO_DIR = '/opt/perfact/dbutils-zoperepo'
DBMODS_POOL = '__root__/PerFact/DatabaseModifications/pool/'
DNG_SQL_PATTERNS = [
    re.compile(r'drop table'),
    re.compile(r'drop column'),
]


def find_dangerous_zsql(repo_dir=REPO_DIR, dbmod_pool=DBMODS_POOL):
    '''
    Find dangerous ZSQL methods containing destructive statements

    repo_dir::    the git repository to scan
                  default /opt/perfact/dbutils-zoperepo

    dbmod_pool::  the subpath in the git repository where the DBMODs live
                  default __root__/PerFact/DatabaseModifications/pool/
    '''

    full_path = os.path.join(repo_dir, dbmod_pool)

    dangerous = []

    for root, dirs, files in os.walk(full_path):
        for obj in files:

            # skip meta files
            if obj == '__meta__':
                continue

            # skip non-sql objects
            if not obj.endswith('sql'):
                continue

            # build full path to obj
            obj_path = os.path.join(root, obj)

            # read file content
            with open(obj_path, 'r') as obj_file:
                content = obj_file.read()

                # search content with each pattern
                for pattern in DNG_SQL_PATTERNS:
                    # XXX: memorize the pattern as well!
                    res = pattern.search(content)
                    if res:  # memorize matching objects
                        dangerous.append(
                            os.path.relpath(obj_path, start=repo_dir)
                        )

    return dangerous


def show_git_info(repo_dir, subpath):
    '''
    Show git log infos for given subpath in given repository directory

    repo_dir:: the git repository to scan
               default /opt/perfact/dbutils-zoperepo

    subpath::  the subpath to show git infos for
    '''
    cmd = [
        'git',
        '-C',
        repo_dir,
        'log',
        '--oneline',
        '--',
        subpath
    ]
    retcode, output = safe_syscall(cmd, raisemode=True)

    return output


def binwrapper_zcode_analysis():

    parser = argparse.ArgumentParser(
        description="Analyze dbutils-zoperepo",
    )
    parser.add_argument(
        'mode',
        type=str,
        choices=['dngr_zsql'],
        help='Analysis mode',
    )

    parser.add_argument(
        '--repo', '-r',
        type=str,
        default=REPO_DIR,
        help='Path to dbutils-zoperepo. Default: %s' % REPO_DIR,
    )
    parser.add_argument(
        '--subpath', '-s',
        type=str,
        default=DBMODS_POOL,
        help='Subpath in dbutils-zoperepo to scan. Default: %s' % DBMODS_POOL,
    )
    parser.add_argument(
        '--show-git-info', '-g',
        default=False,
        action='store_true',
        help='Show additional information from git repo.'
        ' This may take some time',
    )

    args = parser.parse_args()

    if args.mode == 'dngr_zsql':
        res = find_dangerous_zsql(
            repo_dir=args.repo,
            dbmod_pool=args.subpath,
        )

        if res:
            print('Found dangerous DB-Mods:')
            for item in res:
                print(item)
                print()

                if args.show_git_info:
                    git_info = show_git_info(args.repo, item)
                    print(git_info)

                print('------------------------------')
